package streamlog

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"time"

	"code.justin.tv/video/streamlog/pkg/query"
	"github.com/sirupsen/logrus"
)

type Client interface {
	GetLowres(channel string, start, end time.Time) (*query.QueryResultV2, error)
	GetSession(channel, sessionID string) (*query.QueryResultV2, error)
	GetHighRes(channel, sessionID, metric string, start, end time.Time) (*query.QueryResultV2, error)
}

type client struct {
	httpClient        *http.Client
	streamlogEndpoint string
	logger            logrus.FieldLogger
}

func New(streamlogEndpoint string, logger logrus.FieldLogger) Client {
	return &client{
		streamlogEndpoint: streamlogEndpoint,
		httpClient:        &http.Client{},
		logger:            logger,
	}
}

func (c *client) makeStreamlogRequest(req query.QueryRequestV2) (*query.QueryResultV2, error) {
	buf := new(bytes.Buffer)
	err := json.NewEncoder(buf).Encode(req)
	if err != nil {
		return nil, err
	}

	resp, err := c.httpClient.Post(c.streamlogEndpoint+"/api/v2/query", "application/json; charset=utf-8", buf)

	if err != nil {
		return nil, err
	}

	if resp.StatusCode == http.StatusNotFound {
		return nil, nil
	}

	if resp.StatusCode != http.StatusOK {
		c.logger.Warnf("Error from streamlog with request payload:\n%v", buf.String())
		return nil, fmt.Errorf("Non 200 status code from streamlog %v", resp.StatusCode)
	}

	streamlogResponse := &query.QueryResponseV2{}
	defer resp.Body.Close()

	err = json.NewDecoder(resp.Body).Decode(streamlogResponse)
	if err != nil {
		return nil, err
	}

	if streamlogResponse.Code == 200 {
		return streamlogResponse.Result, nil
	}
	if streamlogResponse.Code == 404 {
		return nil, nil
	}

	return nil, fmt.Errorf("failed to get sesssions, error code %v", streamlogResponse.Code)
}

func (c *client) GetSession(channel, sessionID string) (*query.QueryResultV2, error) {
	requestPayload := query.QueryRequestV2{
		Channel:   channel,
		SessionID: sessionID,
		Type:      "lowres",
		Stop:      time.Now().Unix(),
	}

	return c.makeStreamlogRequest(requestPayload)
}

func (c *client) GetLowres(channel string, start, end time.Time) (*query.QueryResultV2, error) {
	requestPayload := query.QueryRequestV2{
		Channel: channel,
		Type:    "lowres",
		Start:   start.Unix(),
		Stop:    end.Unix(),
	}

	return c.makeStreamlogRequest(requestPayload)
}

func (c *client) GetHighRes(channel, sessionID, metric string, start, end time.Time) (*query.QueryResultV2, error) {
	requestPayload := query.QueryRequestV2{
		Channel:   channel,
		SessionID: sessionID,
		Metric:    metric,
		Type:      "highres",
		Start:     start.Unix(),
		Stop:      end.Unix(),
	}

	return c.makeStreamlogRequest(requestPayload)
}
