package svc

import (
	"context"
	"encoding/json"
	"fmt"
	"strings"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/golang/protobuf/ptypes"
	"github.com/twitchtv/twirp"

	mienc "code.justin.tv/event-engineering/carrot-stream-analysis/pkg/mediainfo/encoding"
	csa "code.justin.tv/event-engineering/carrot-stream-analysis/pkg/rpc"
)

func (c *client) GetCapturedFileDetail(ctx context.Context, request *csa.GetCapturedFileDetailRequest) (*csa.GetCapturedFileDetailResponse, error) {
	// Make sure the original file exists
	_, err := c.s3.HeadObject(&s3.HeadObjectInput{
		Bucket: aws.String(c.flvAnalyserBucketName),
		Key:    aws.String(request.Key),
	})

	if err != nil {
		c.logger.WithError(err).Warnf("Failed to perform HeadObject on file %v/%v", c.flvAnalyserBucketName, request.Key)
		return nil, err
	}

	// Grab the basic analysis
	basicErr := false
	basic, err := c.s3.GetObject(&s3.GetObjectInput{
		Bucket: aws.String(c.flvAnalyserBucketName),
		Key:    aws.String(fmt.Sprintf("%v.basic.json", request.Key)),
	})

	if err != nil {
		basicErr = true
		c.logger.WithError(err).Warnf("Failed to retrieve basic analysis for file %v", request.Key)
	}

	// Grab the GOP analysis
	gopErr := false
	gop, err := c.s3.GetObject(&s3.GetObjectInput{
		Bucket: aws.String(c.flvAnalyserBucketName),
		Key:    aws.String(fmt.Sprintf("%v.gop.json", request.Key)),
	})

	if err != nil {
		gopErr = true
		c.logger.WithError(err).Warnf("Failed to retrieve GOP analysis for file %v", request.Key)
	}

	resp := &csa.GetCapturedFileDetailResponse{}

	if !basicErr {
		basicDetail := &mienc.BasicDetail{}
		dec := json.NewDecoder(basic.Body)

		err = dec.Decode(basicDetail)
		if err != nil {
			c.logger.WithError(err).Warnf("Failed to unmarshal basic detail to JSON %v", request.Key)
		} else if basicDetail.Media != nil {
			resp.Detail = &csa.CapturedFileDetail{}

			// The way mediainfo does JSON is.... weird
			for _, track := range basicDetail.Media.Tracks {
				if track.Type == "General" {
					resp.Detail.Size = c.parseInt64(track.FileSize)
					resp.Detail.Duration = c.parseFloat64(track.Duration)
					resp.Detail.OverallBitrate = c.parseInt64(track.OverallBitRate)
					resp.Detail.Framerate = c.parseFloat64(track.FrameRate)
					resp.Detail.NumVideoTracks = c.parseInt32(track.VideoCount)
					resp.Detail.NumAudioTracks = c.parseInt32(track.AudioCount)

					continue
				}

				if track.Type == "Video" {
					resp.Detail.VideoDetail = &csa.CapturedFileVideoDetail{
						Format:                track.Format,
						Profile:               track.FormatProfile,
						Level:                 c.parseFloat64(track.FormatLevel),
						Cabac:                 track.FormatSettingsCABAC == "Yes",
						RefFrames:             c.parseInt32(track.FormatSettingsRefFrames),
						CodecId:               c.parseInt32(track.CodecID),
						Bitrate:               c.parseInt64(track.BitRate),
						BitrateNominal:        c.parseInt64(track.BitRateNominal),
						Width:                 c.parseInt32(track.Width),
						WidthSampled:          c.parseInt32(track.SampledWidth),
						Height:                c.parseInt32(track.Height),
						HeightSampled:         c.parseInt32(track.SampledHeight),
						PixelAspectRatio:      c.parseFloat64(track.PixelAspectRatio),
						DisplayAspectRatio:    c.parseFloat64(track.DisplayAspectRatio),
						FramerateMode:         track.FrameRateMode,
						FramerateModeOriginal: track.FrameRateModeOriginal,
						Framerate:             c.parseFloat64(track.FrameRate),
						FramerateOriginal:     c.parseFloat64(track.FrameRateOriginal),
						FrameCount:            c.parseInt64(track.FrameCount),
						ColourSpace:           track.ColorSpace, // RULE BRITANNIA
						ChromaSubsampling:     track.ChromaSubsampling,
						BitDepth:              c.parseInt32(track.BitDepth),
						ScanType:              track.ScanType,
						Delay:                 c.parseFloat64(track.Delay),
						EncoderLibrary:        track.EncodedLibrary,
						EncoderSettings:       make(map[string]string, 0),
					}

					for _, setting := range strings.Split(track.EncodedLibrarySettings, " / ") {
						parts := strings.Split(setting, "=")
						if len(parts) != 2 {
							continue
						}

						resp.Detail.VideoDetail.EncoderSettings[parts[0]] = parts[1]
					}

					continue
				}

				if track.Type == "Audio" {
					resp.Detail.AudioDetail = &csa.CapturedFileAudioDetail{
						Format:             track.Format,
						Sbr:                track.FormatSettingsSBR,
						AdditionalFeatures: track.FormatAdditionalFeatures,
						CodecId:            track.CodecID,
						Bitrate:            c.parseInt64(track.BitRate),
						Channels:           c.parseInt32(track.Channels),
						ChannelPositions:   track.ChannelPositions,
						ChannelLayout:      track.ChannelLayout,
						SamplesPerFrame:    c.parseInt32(track.SamplesPerFrame),
						SampleRate:         c.parseInt32(track.SamplingRate),
						SampleCount:        c.parseInt64(track.SamplingCount),
						Framerate:          c.parseFloat64(track.FrameRate),
						CompressionMode:    track.CompressionMode,
						Delay:              c.parseFloat64(track.Delay),
						DelaySource:        track.DelaySource,
					}
					continue
				}
			}
		}
	}

	if !gopErr {
		resp.Gop = &csa.CapturedFileGOP{}

		gopData := &mienc.GOPData{}
		dec := json.NewDecoder(gop.Body)

		err = dec.Decode(gopData)
		if err != nil {
			c.logger.WithError(err).Warnf("Failed to unmarshal GOP data to JSON %v", request.Key)
		} else if gopData.Frames != nil {
			resp.Gop.Frames = make([]*csa.GOPInfoFrame, 0)

			for _, frame := range gopData.Frames {
				resp.Gop.Frames = append(resp.Gop.Frames, &csa.GOPInfoFrame{
					Timestamp: c.parseInt64(frame.Timestamp),
					PictType:  frame.PictType,
					Size:      c.parseInt64(frame.Size),
				})
			}
		}
	}

	return resp, nil
}

func (c *client) ListCapturedFiles(context context.Context, request *csa.ListCapturedFilesRequest) (*csa.ListCapturedFilesResponse, error) {
	resp, err := c.s3.ListObjectsV2(&s3.ListObjectsV2Input{
		Bucket: aws.String(c.flvAnalyserBucketName),
		Prefix: aws.String(request.Prefix),
	})

	if err != nil {
		c.logger.WithError(err).Warnf("Failed to get recorded files with prefix %v", request.Prefix)
		return nil, twirp.InternalError("Error accessing S3")
	}

	files := make([]*csa.CapturedFileInfo, 0, len(resp.Contents))

	for _, file := range resp.Contents {
		if !strings.HasSuffix(*file.Key, ".flv") {
			continue
		}

		lastModified, err := ptypes.TimestampProto(*file.LastModified)
		if err != nil {
			c.logger.WithError(err).Warnf("Failed to convert last modified from time.Time to protobuf Timestamp: %v", *file.Key)
		}

		files = append(files, &csa.CapturedFileInfo{
			Key:          *file.Key,
			LastModified: lastModified,
			Size:         *file.Size,
		})
	}

	return &csa.ListCapturedFilesResponse{
		Files: files,
	}, nil
}

func (c *client) GetCapturedFileDownloadLink(ctx context.Context, request *csa.GetCapturedFileDownloadLinkRequest) (*csa.GetCapturedFileDownloadLinkResponse, error) {
	if !strings.HasSuffix(request.Key, ".flv") {
		return nil, twirp.InvalidArgumentError("key", "Only FLV files can be downloaded")
	}

	dlReq, _ := c.s3.GetObjectRequest(&s3.GetObjectInput{
		Bucket: aws.String(c.flvAnalyserBucketName),
		Key:    aws.String(request.Key),
	})

	downloadLink, err := dlReq.Presign(1 * time.Minute)

	if err != nil {
		return nil, twirp.InternalError("Failed to generate presigned URL")
	}

	return &csa.GetCapturedFileDownloadLinkResponse{
		DownloadLink: downloadLink,
	}, nil
}
