package svc

import (
	"encoding/json"
	"errors"
	"math"
	"regexp"
	"strconv"
	"strings"
	"time"

	csa "code.justin.tv/event-engineering/carrot-stream-analysis/pkg/rpc"
	"code.justin.tv/video/streamlog/pkg/query"
	pDuration "github.com/golang/protobuf/ptypes/duration"
	pTimestamp "github.com/golang/protobuf/ptypes/timestamp"
	"github.com/sirupsen/logrus"
)

func createSessionInfoList(records map[string]*query.QueryRecords, logger logrus.FieldLogger) map[string]*csa.SessionInfo {
	sessions := map[string]*csa.SessionInfo{}

	for metric, records := range records {
		for _, record := range records.Recs {
			res, ok := sessions[record.SessionID]
			if !ok {
				sessions[record.SessionID] = &csa.SessionInfo{
					SessionId:       record.SessionID,
					RelatedSessions: make([]*csa.SessionInfo, 0),
				}
				res = sessions[record.SessionID]
			}
			err := addSessionInfoMetric(res, metric, record)
			if err != nil {
				logger.WithField("Error", err).Error("Error adding session info metric")
			}
		}
	}

	return sessions
}

func addSessionInfoMetric(si *csa.SessionInfo, metric string, record *query.QueryRecord) error {
	switch metric {
	case "stream_up":
		si.Start = &pTimestamp.Timestamp{
			Seconds: record.Timestamp,
		}
		break
	case "stream_down":
		si.End = &pTimestamp.Timestamp{
			Seconds: record.Timestamp,
		}
		break
	case "ingest_proxy":
		val, err := getString(record)
		if err != nil {
			return err
		}
		si.IngestEdge = val
		break
	case "broadcast_format":
		val, err := getString(record)
		if err != nil {
			return err
		}
		si.BroadcastFormat = val
		break
	}

	return nil
}

func getString(qr *query.QueryRecord) (string, error) {
	if val, ok := qr.Value.(string); ok {
		return val, nil
	}
	return "", errors.New("Value was not of type string")
}

var ingestEdgeProxyRegexp = regexp.MustCompile(`.([a-z]{3})[0-9]{2}.justin.tv`)

func extractIngestServerProxy(host string) string {
	matches := ingestEdgeProxyRegexp.FindStringSubmatch(host)
	if len(matches) != 2 {
		return ""
	}

	return strings.ToUpper(matches[1])
}

func extractTimestamp(sessionID, key string, recordMap map[string]*query.QueryRecords) int64 {
	recs, ok := recordMap[key]
	if !ok {
		return 0
	}
	for _, rec := range recs.Recs {
		if rec.SessionID == sessionID && rec.Timestamp > 0 {
			return rec.Timestamp
		}
	}
	return 0
}

func extractStringValue(sessionID, key string, recordMap map[string]*query.QueryRecords) string {
	recs, ok := recordMap[key]
	if !ok {
		return ""
	}

	for _, rec := range recs.Recs {
		if rec.SessionID != sessionID {
			continue
		}

		if val, ok := rec.Value.(string); ok {
			return val
		}
	}
	return ""
}

func extractStringIntValue(sessionID, key string, recordMap map[string]*query.QueryRecords) int64 {
	recs, ok := recordMap[key]
	if !ok {
		return 0
	}

	for _, rec := range recs.Recs {
		if rec.SessionID != sessionID {
			continue
		}

		if val, ok := rec.Value.(string); ok {
			intVal, err := strconv.ParseInt(val, 10, 64)
			if err != nil {
				return 0
			}
			return intVal
		}
	}
	return 0
}

func extractFloatIntValue(sessionID, key string, recordMap map[string]*query.QueryRecords) int64 {
	recs, ok := recordMap[key]
	if !ok {
		return 0
	}

	for _, rec := range recs.Recs {
		if rec.SessionID != sessionID {
			continue
		}

		if val, ok := rec.Value.(float64); ok {
			return int64(val)
		}
	}

	return 0
}

func extractIntValue(sessionID, key string, recordMap map[string]*query.QueryRecords) int64 {
	recs, ok := recordMap[key]
	if !ok {
		return 0
	}

	for _, rec := range recs.Recs {
		if rec.SessionID != sessionID {
			continue
		}

		if val, ok := rec.Value.(int); ok {
			return int64(val)
		}

		if val, ok := rec.Value.(int32); ok {
			return int64(val)
		}

		if val, ok := rec.Value.(int64); ok {
			return int64(val)
		}
	}

	return 0
}

type FloatValueWithTimestamp struct {
	Timestamp time.Time
	Value     float64
}

func extractFloatValuesWithTimestamp(sessionID, key string, recordMap map[string]*query.QueryRecords) []*FloatValueWithTimestamp {
	recs, ok := recordMap[key]
	if !ok {
		return []*FloatValueWithTimestamp{}
	}

	var result []*FloatValueWithTimestamp
	for _, rec := range recs.Recs {
		if rec.SessionID != sessionID {
			continue
		}

		if val, ok := rec.Value.(float64); ok {
			result = append(result, &FloatValueWithTimestamp{Timestamp: time.Unix(rec.Timestamp, 0), Value: val})
			continue
		}

		if val, ok := rec.Value.(float32); ok {
			result = append(result, &FloatValueWithTimestamp{Timestamp: time.Unix(rec.Timestamp, 0), Value: float64(val)})
			continue
		}
	}
	return result
}

type DurationWithTimestamp struct {
	Timestamp time.Time
	Duration  time.Duration
}

// For duration records like starvation or dropped frame on Streamlog, the start and end of each event is recorded `true` and `false`
// with a unix timestamp. For the record that start with true but doesn't end with false, this function considers the event as a
// 0 second duration event.
func extractDurationWithTimestamp(sessionID, key string, recordMap map[string]*query.QueryRecords) []*DurationWithTimestamp {
	recs, ok := recordMap[key]
	if !ok {
		return []*DurationWithTimestamp{}
	}

	var result []*DurationWithTimestamp
	for _, rec := range recs.Recs {
		if rec.SessionID != sessionID {
			continue
		}

		val, ok := rec.Value.(bool)
		if !ok {
			continue
		}

		if val {
			result = append(result, &DurationWithTimestamp{
				Timestamp: time.Unix(rec.Timestamp, 0),
			})
			continue
		}

		if !val && len(result) > 0 && result[len(result)-1].Duration == 0 {
			end := time.Unix(rec.Timestamp, 0)
			result[len(result)-1].Duration = end.Sub(result[len(result)-1].Timestamp)
			continue
		}
	}
	return result
}

type StringValueWithTimestamp struct {
	Timestamp time.Time
	Value     string
}

func extractStringValuesWithTimestamp(sessionID, key string, recordMap map[string]*query.QueryRecords) []*StringValueWithTimestamp {
	recs, ok := recordMap[key]
	if !ok {
		return []*StringValueWithTimestamp{}
	}

	var result []*StringValueWithTimestamp
	for _, rec := range recs.Recs {
		if rec.SessionID != sessionID {
			continue
		}

		if val, ok := rec.Value.(string); ok {
			result = append(result, &StringValueWithTimestamp{Timestamp: time.Unix(rec.Timestamp, 0), Value: val})
			continue
		}
	}
	return result
}

type FloatIntValueWithTimestamp struct {
	Timestamp time.Time
	Value     int64
}

func extractFloatIntValuesWithTimestamp(sessionID, key string, recordMap map[string]*query.QueryRecords) []*FloatIntValueWithTimestamp {
	recs, ok := recordMap[key]
	if !ok {
		return []*FloatIntValueWithTimestamp{}
	}

	var result []*FloatIntValueWithTimestamp
	for _, rec := range recs.Recs {
		if rec.SessionID != sessionID {
			continue
		}

		if val, ok := rec.Value.(float64); ok {
			result = append(result, &FloatIntValueWithTimestamp{Timestamp: time.Unix(rec.Timestamp, 0), Value: int64(val)})
			continue
		}
	}
	return result
}

type StitchSource struct {
	SessionID string
	Format    string
	Channel   string
	ChannelID string
	RtmpURL   string
	Timestamp int64
}

func extractBackupSession(sessionID, metric string, records map[string]*query.QueryRecords) []*StitchSource {
	stitchSourceIDs := []*StitchSource{}
	for key, value := range records {
		if key != metric {
			continue
		}

		for _, rec := range value.Recs {
			if rec.SessionID != sessionID {
				continue
			}

			stitch, ok := rec.Value.(string)
			if !ok {
				continue
			}

			var stitchSource *StitchSource
			if err := json.Unmarshal([]byte(stitch), &stitchSource); err != nil {
				continue
			}
			stitchSource.Timestamp = rec.Timestamp

			stitchSourceIDs = append(stitchSourceIDs, stitchSource)
		}
	}
	return stitchSourceIDs
}

func extractSessionIDs(key string, recordMap map[string]*query.QueryRecords) []string {
	recs, ok := recordMap[key]
	if !ok || recs == nil {
		return []string{}
	}

	// use counter to keep track of order since iterating key, value loses the order
	counter := 0
	sessionIDMap := make(map[string]int)
	for _, rec := range recs.Recs {
		if rec.SessionID == "" {
			continue
		}
		if _, ok := sessionIDMap[rec.SessionID]; !ok {
			sessionIDMap[rec.SessionID] = counter
			counter++
		}
	}

	sessionIDs := make([]string, counter)
	for k, v := range sessionIDMap {
		sessionIDs[v] = k
	}
	return sessionIDs
}

func mapIngestSession(sessionID string, records map[string]*query.QueryRecords) *csa.IngestSession {
	session := &csa.IngestSession{}

	session.BroadcastFormat = extractStringValue(sessionID, "broadcast_format", records)
	if timestamp := extractTimestamp(sessionID, "stream_up", records); timestamp > 0 {
		session.StreamUp = &pTimestamp.Timestamp{Seconds: timestamp}
	}
	if timestamp := extractTimestamp(sessionID, "stream_down", records); timestamp > 0 {
		session.StreamDown = &pTimestamp.Timestamp{Seconds: timestamp}
	}
	session.IngestHost = extractStringValue(sessionID, "ingest_proxy", records)
	session.IngestProxy = extractIngestServerProxy(session.IngestHost)
	session.Delay = &pDuration.Duration{Seconds: extractFloatIntValue(sessionID, "delay", records)}

	return session
}

func mapRTMPSession(sessionID string, records map[string]*query.QueryRecords) *csa.RTMPSession {
	session := &csa.RTMPSession{}

	session.AudioCodecs = extractStringValue(sessionID, "source_audio_codec", records)
	session.VideoCodecs = extractStringValue(sessionID, "source_video_codec", records)
	session.VideoResolutionHeight = extractStringIntValue(sessionID, "video_height", records)
	session.VideoResolutionWidth = extractStringIntValue(sessionID, "video_width", records)
	session.AvcLevel = extractFloatIntValue(sessionID, "avc_level", records)
	session.IdrInterval = extractFloatIntValue(sessionID, "source_idr_interval", records)
	session.SegmentDuration = extractFloatIntValue(sessionID, "segment_duration", records)
	session.RtmpMetadata = extractStringValue(sessionID, "rtmp_metadata", records)
	session.RtmpFlags = extractStringValue(sessionID, "rtmp_flags", records)
	session.RtmpExitReason = extractStringValue(sessionID, "rtmpexit_reason", records)
	session.Encoder = extractStringValue(sessionID, "encoder", records)
	session.ClientIp = extractStringValue(sessionID, "client_ip", records)
	session.VideoDataRate = extractStringIntValue(sessionID, "video_data_rate", records)
	session.VideoFrameRate = extractStringIntValue(sessionID, "video_framerate", records)
	session.AudioSampleRate = extractStringIntValue(sessionID, "audio_sample_rate", records)
	session.AudioDataRate = extractStringIntValue(sessionID, "audio_data_rate", records)
	session.AudioChannels = extractStringIntValue(sessionID, "audio_channels", records)

	strStereo := extractStringValue(sessionID, "audio_stereo", records)
	strBool, err := strconv.ParseBool(strStereo)
	if err == nil {
		session.AudioIsStereo = strBool
	}

	session.AudioSampleSize = extractStringIntValue(sessionID, "audio_sample_size", records)

	return session
}

func mapTranscodeSessions(sessionID string, records map[string]*query.QueryRecords) []*csa.TranscodeSession {
	transcodeHosts := extractStringValuesWithTimestamp(sessionID, "transcode_host", records)
	transcodeSessions := make([]*csa.TranscodeSession, 0, len(transcodeHosts))
	for _, event := range transcodeHosts {
		transcodeSessions = append(transcodeSessions, &csa.TranscodeSession{
			Host:      event.Value,
			Timestamp: &pTimestamp.Timestamp{Seconds: event.Timestamp.Unix()},
		})
	}
	profiles := extractStringValuesWithTimestamp(sessionID, "transcode_profile", records)
	for i, event := range profiles {
		if i > len(transcodeSessions)-1 || transcodeSessions[i] == nil || transcodeSessions[i].Timestamp == nil {
			continue
		}
		if math.Abs(float64(event.Timestamp.Unix()-transcodeSessions[i].Timestamp.Seconds)) > 60 {
			continue
		}

		transcodeSessions[i].Profile = event.Value
	}
	origins := extractStringValuesWithTimestamp(sessionID, "origin", records)
	for i, event := range origins {
		if i > len(transcodeSessions)-1 || transcodeSessions[i] == nil || transcodeSessions[i].Timestamp == nil {
			continue
		}
		if math.Abs(float64(event.Timestamp.Unix()-transcodeSessions[i].Timestamp.Seconds)) > 60 {
			continue
		}

		transcodeSessions[i].IngestOrigin = event.Value
	}
	audioCodecs := extractStringValuesWithTimestamp(sessionID, "audio_codec", records)
	for i, event := range audioCodecs {
		if i > len(transcodeSessions)-1 || transcodeSessions[i] == nil || transcodeSessions[i].Timestamp == nil {
			continue
		}
		if math.Abs(float64(event.Timestamp.Unix()-transcodeSessions[i].Timestamp.Seconds)) > 60 {
			continue
		}

		transcodeSessions[i].AudioCodecs = event.Value
	}
	videoCodecs := extractStringValuesWithTimestamp(sessionID, "video_codec", records)
	for i, event := range videoCodecs {
		if i > len(transcodeSessions)-1 || transcodeSessions[i] == nil || transcodeSessions[i].Timestamp == nil {
			continue
		}
		if math.Abs(float64(event.Timestamp.Unix()-transcodeSessions[i].Timestamp.Seconds)) > 60 {
			continue
		}

		transcodeSessions[i].VideoCodecs = event.Value
	}
	stackVersion := extractFloatIntValuesWithTimestamp(sessionID, "transcode_stack_version", records)
	for i, event := range stackVersion {
		if i > len(transcodeSessions)-1 || transcodeSessions[i] == nil || transcodeSessions[i].Timestamp == nil {
			continue
		}
		if math.Abs(float64(event.Timestamp.Unix()-transcodeSessions[i].Timestamp.Seconds)) > 60 {
			continue
		}

		transcodeSessions[i].StackVersion = event.Value
	}
	recordingPrefix := extractStringValuesWithTimestamp(sessionID, "recording_prefix", records)
	for i, event := range recordingPrefix {
		if i > len(transcodeSessions)-1 || transcodeSessions[i] == nil || transcodeSessions[i].Timestamp == nil {
			continue
		}
		if math.Abs(float64(event.Timestamp.Unix()-transcodeSessions[i].Timestamp.Seconds)) > 60 {
			continue
		}

		transcodeSessions[i].RecordingPrefix = event.Value
	}
	recordingS3Bucket := extractStringValuesWithTimestamp(sessionID, "recording_s3_bucket", records)
	for i, event := range recordingS3Bucket {
		if i > len(transcodeSessions)-1 || transcodeSessions[i] == nil || transcodeSessions[i].Timestamp == nil {
			continue
		}
		if math.Abs(float64(event.Timestamp.Unix()-transcodeSessions[i].Timestamp.Seconds)) > 60 {
			continue
		}

		transcodeSessions[i].RecordingS3Bucket = event.Value
	}
	return transcodeSessions
}

func mapBackupSession(sessionID string, records map[string]*query.QueryRecords) *csa.BackupIngestSession {
	session := &csa.BackupIngestSession{}

	if stitchedSessions := extractBackupSession(sessionID, "stitch_source", records); len(stitchedSessions) > 0 {
		for _, stitchedSession := range stitchedSessions {
			session.StitchedTo = append(session.StitchedTo, &csa.BackupIngestSession_BackupSession{
				SessionId:         stitchedSession.SessionID,
				BroadcastFormat:   stitchedSession.Format,
				StitchedTimestamp: &pTimestamp.Timestamp{Seconds: stitchedSession.Timestamp},
			})
		}
	}

	if stitchedSessions := extractBackupSession(sessionID, "stitch_from", records); len(stitchedSessions) > 0 {
		for _, stitchedSession := range stitchedSessions {
			session.StitchedFrom = append(session.StitchedFrom, &csa.BackupIngestSession_BackupSession{
				SessionId:         stitchedSession.SessionID,
				BroadcastFormat:   stitchedSession.Format,
				StitchedTimestamp: &pTimestamp.Timestamp{Seconds: stitchedSession.Timestamp},
			})
		}
	}

	return session
}

func mapStarvations(sessionID string, records map[string]*query.QueryRecords) []*csa.Starvation {
	starvationRecords := extractDurationWithTimestamp(sessionID, "starved", records)
	starvations := make([]*csa.Starvation, len(starvationRecords))
	for i, starvation := range starvationRecords {
		starvations[i] = &csa.Starvation{
			Start:    &pTimestamp.Timestamp{Seconds: starvation.Timestamp.Unix()},
			Duration: &pDuration.Duration{Seconds: int64(starvation.Duration.Seconds())},
		}
	}

	return starvations
}

func mapFrameDrops(sessionID string, records map[string]*query.QueryRecords) []*csa.FrameDrop {
	dropedFrames := extractDurationWithTimestamp(sessionID, "framesdropped", records)
	frameDrops := make([]*csa.FrameDrop, len(dropedFrames))
	for i, dropedFrame := range dropedFrames {
		frameDrops[i] = &csa.FrameDrop{
			Start:    &pTimestamp.Timestamp{Seconds: dropedFrame.Timestamp.Unix()},
			Duration: &pDuration.Duration{Seconds: int64(dropedFrame.Duration.Seconds())},
		}
	}
	return frameDrops
}

func (c *client) parseInt64(in string) int64 {
	if in == "" {
		return 0
	}

	out, err := strconv.ParseInt(in, 10, 64)
	if err != nil {
		c.logger.WithError(err).Warnf("Unable to convert string to int64 %v", in)
		return 0
	}

	return out
}

func (c *client) parseFloat64(in string) float64 {
	if in == "" {
		return 0
	}

	out, err := strconv.ParseFloat(in, 64)
	if err != nil {
		c.logger.WithError(err).Warnf("Unable to convert string to float64 %v", in)
		return 0
	}

	return out
}

func (c *client) parseInt32(in string) int32 {
	if in == "" {
		return 0
	}

	out, err := strconv.ParseFloat(in, 32)
	if err != nil {
		c.logger.WithError(err).Warnf("Unable to convert string to int32 %v", in)
		return 0
	}

	return int32(out)
}
