package svc

import (
	"context"
	"fmt"
	"sort"
	"time"

	"code.justin.tv/event-engineering/carrot-stream-analysis/pkg/mmdb"
	csa "code.justin.tv/event-engineering/carrot-stream-analysis/pkg/rpc"
	"code.justin.tv/event-engineering/carrot-stream-analysis/pkg/streamlog"
	"github.com/aws/aws-sdk-go/service/s3/s3iface"
	"github.com/golang/protobuf/ptypes"
	pTimestamp "github.com/golang/protobuf/ptypes/timestamp"
	"github.com/sirupsen/logrus"
	"github.com/twitchtv/twirp"
)

/*
	A lot of this is lifted from velen and will do for now, we should come back and tidy this up when we have time
*/

// Client defines the functions that will be available in this service, in this case it's pretty much a straight implementation of the twirp service
type Client interface {
	GetChannelSessions(context context.Context, request *csa.GetChannelSessionsRequest) (*csa.GetChannelSessionsResponse, error)
	GetSessionData(context context.Context, request *csa.GetSessionDataRequest) (*csa.GetSessionDataResponse, error)
	GetCapturedFileDetail(ctx context.Context, request *csa.GetCapturedFileDetailRequest) (*csa.GetCapturedFileDetailResponse, error)
	ListCapturedFiles(ctx context.Context, request *csa.ListCapturedFilesRequest) (*csa.ListCapturedFilesResponse, error)
	GetCapturedFileDownloadLink(ctx context.Context, request *csa.GetCapturedFileDownloadLinkRequest) (*csa.GetCapturedFileDownloadLinkResponse, error)
}

type client struct {
	streamlogEndpoint     string
	streamlog             streamlog.Client
	mmdb                  mmdb.Client
	s3                    s3iface.S3API
	flvAnalyserBucketName string
	logger                logrus.FieldLogger
}

// New returns a new Carrot Stream Analysis client
func New(streamlogEndpoint string, mmdbClient mmdb.Client, s3Client s3iface.S3API, flvAnalyserBucketName string, logger logrus.FieldLogger) Client {
	streamlogClient := streamlog.New(streamlogEndpoint, logger)

	return &client{
		streamlogEndpoint:     streamlogEndpoint,
		streamlog:             streamlogClient,
		mmdb:                  mmdbClient,
		s3:                    s3Client,
		flvAnalyserBucketName: flvAnalyserBucketName,
		logger:                logger,
	}
}

func (c *client) GetChannelSessions(context context.Context, request *csa.GetChannelSessionsRequest) (*csa.GetChannelSessionsResponse, error) {
	slChannel := formatStreamlogChannel(request.CustomerId, request.Region, request.ContentId)
	logger := c.logger.WithFields(logrus.Fields{
		"Endpoint": "GetChannelSessions",
	})

	start, err := ptypes.Timestamp(request.Start)
	if err != nil {
		logger.WithField("Error", err).Error("Failed to convert request.Start to go time.Time")
		return nil, twirp.InvalidArgumentError("start", "Failed to convert request.Start to go time.Time")
	}

	end, err := ptypes.Timestamp(request.End)
	if err != nil {
		logger.WithField("Error", err).Error("Failed to convert request.End to go time.Time")
		return nil, twirp.InvalidArgumentError("end", "Failed to convert request.End to go time.Time")
	}

	// TODO validate these timestamps are within a certain range? Not sure what that should be just yet

	logger.Debugf("Attempting LowRes query for %v", slChannel)
	resp, err := c.streamlog.GetLowres(slChannel, start, end)
	if err != nil {
		logger.WithField("Error", err).Error("Failed to retrieve data from streamlog")
		return nil, twirp.InternalError("Failed to retrieve session list")
	}

	if resp == nil {
		return nil, twirp.NotFoundError("No session data found")
	}

	if resp.Channel != slChannel {
		logger.Errorf("Streamlog channel does not equal request channel %v vs. %v", resp.Channel, slChannel)
		return nil, twirp.InternalError("Streamlog channel mismatch")
	}

	twirpResponse := &csa.GetChannelSessionsResponse{
		Sessions: make([]*csa.SessionInfo, 0),
	}

	// We're using a map while we populate the data to make it easier to reference the sessions by ID
	sessions := createSessionInfoList(resp.Records, logger)

	// We're going to return backup/slate sessions as part of the master "live" session rather than listing them independently so we need to fix them up here
	// There's no hard link between primary and backup sessions until a failover event occurs, so we're just going to list the "backup.00x" sessions that are streaming or have streamed
	// at any point to the same customer/content id between the start / end timestamps of the master "live" session
	relatedSessions := make([]*csa.SessionInfo, 0)

	// Now that we've mapped the reponse from streamlog into sensible objects, add them to the list
	for _, session := range sessions {
		if session.Start == nil {
			// This can happen when the stream started before [start] but ended after, so there are some events for it
			// In this scenario, since is started before [start] we're just not going to return it
			continue
		}

		// Workaround for https://jira.twitch.com/browse/ING-6113
		if session.End == nil {
			goStart, err := ptypes.Timestamp(session.Start)
			if err == nil && time.Now().Sub(goStart) > time.Hour*48 {
				goEnd := goStart.Add(time.Hour * 48)
				pEnd, err := ptypes.TimestampProto(goEnd)
				if err == nil {
					session.End = pEnd
				}
			}
		}

		if session.BroadcastFormat == "live" {
			twirpResponse.Sessions = append(twirpResponse.Sessions, session)
		} else {
			relatedSessions = append(relatedSessions, session)
		}
	}

	if len(relatedSessions) > 0 {
		now := *ptypes.TimestampNow()
		for _, session := range twirpResponse.Sessions {
			for _, rSession := range relatedSessions {
				end := now
				rEnd := now
				if session.End != nil {
					end = *session.End
				}
				if rSession.End != nil {
					rEnd = *rSession.End
				}

				if rSession.Start.Seconds > session.Start.Seconds && rEnd.Seconds <= end.Seconds {
					session.RelatedSessions = append(session.RelatedSessions, rSession)
				}
			}
		}
	}

	// Sort the results by stream up time
	sort.Slice(twirpResponse.Sessions, func(i, j int) bool {
		return twirpResponse.Sessions[i].Start.Seconds > twirpResponse.Sessions[j].Start.Seconds
	})

	return twirpResponse, nil
}

func (c *client) GetSessionData(context context.Context, request *csa.GetSessionDataRequest) (*csa.GetSessionDataResponse, error) {
	slChannel := formatStreamlogChannel(request.CustomerId, request.Region, request.ContentId)
	logger := c.logger.WithFields(logrus.Fields{
		"Endpoint": "GetSessionData",
	})

	result, err := c.streamlog.GetSession(slChannel, request.SessionId)
	if err != nil {
		logger.WithField("Error", err).Error("Failed to retrieve session data streamlog")
		return nil, twirp.InternalError("Failed to retrieve session")
	}

	// 404 from streamlog will result in this
	if result == nil {
		return nil, twirp.NotFoundError("Session Not Found")
	}

	session := &csa.Session{
		SessionId:     request.SessionId,
		IngestSession: &csa.IngestSession{},
		RtmpSession:   &csa.RTMPSession{},
	}

	session.IngestSession = mapIngestSession(request.SessionId, result.Records)
	session.RtmpSession = mapRTMPSession(request.SessionId, result.Records)
	session.TranscodeSessions = mapTranscodeSessions(request.SessionId, result.Records)
	session.BackupIngestSession = mapBackupSession(request.SessionId, result.Records)
	session.Starvations = mapStarvations(request.SessionId, result.Records)
	session.FrameDrops = mapFrameDrops(request.SessionId, result.Records)

	var pEnd *pTimestamp.Timestamp

	if session.IngestSession.StreamDown != nil {
		pEnd = session.IngestSession.StreamDown
	} else {
		pEnd = &pTimestamp.Timestamp{Seconds: time.Now().Unix()}
	}

	end, err := ptypes.Timestamp(pEnd)
	if err != nil {
		logger.WithField("Error", err).Error("Failed to parse session end time")
		return nil, twirp.InternalError("Failed to parse session end time")
	}

	start, err := ptypes.Timestamp(session.IngestSession.StreamUp)
	if err != nil {
		logger.WithField("Error", err).Error("Failed to parse session start time")
		return nil, twirp.InternalError("Failed to parse session end time")
	}

	frResp, err := c.streamlog.GetHighRes(slChannel, request.SessionId, "frame_rate", start, end)
	if err != nil {
		logger.WithField("Error", err).Warnf("Failed to retrieve framerate data from streamlog")
	} else if frResp != nil {
		values := extractFloatValuesWithTimestamp(request.SessionId, "frame_rate", frResp.Records)

		session.Framerates = make([]*csa.Framerate, len(values))

		for i, value := range values {
			session.Framerates[i] = &csa.Framerate{
				ValueFps:  float32(value.Value),
				Timestamp: &pTimestamp.Timestamp{Seconds: value.Timestamp.Unix()},
			}
		}
	}

	brResp, err := c.streamlog.GetHighRes(slChannel, request.SessionId, "bitrate", start, end)
	if err != nil {
		logger.WithField("Error", err).Warnf("Failed to retrieve bitrate data from streamlog")
	} else if brResp != nil {
		values := extractFloatValuesWithTimestamp(request.SessionId, "bitrate", brResp.Records)

		session.Bitrates = make([]*csa.Bitrate, len(values))

		for i, value := range values {
			session.Bitrates[i] = &csa.Bitrate{
				ValueKbps: float32(value.Value),
				Timestamp: &pTimestamp.Timestamp{Seconds: value.Timestamp.Unix()},
			}
		}
	}

	// We're going to return backup/slate sessions as part of the master "live" session rather than listing them independently so we need to go look them up here
	// There's no hard link between primary and backup sessions until a failover event occurs, so we're just going to list the "backup.00x" sessions that are streaming or have streamed
	// at any point to the same customer/content id between the start / end timestamps of the master "live" session
	sessionStart, err := ptypes.Timestamp(session.IngestSession.StreamUp)
	if err != nil {
		logger.WithField("Error", err).Error("Failed to parse Session Start for related session lookup")
		return nil, twirp.InternalError("Failed to parse Session Start for related session lookup")
	}

	sessionEnd := time.Now()

	if session.IngestSession.StreamDown != nil {
		sessionEnd, err = ptypes.Timestamp(session.IngestSession.StreamDown)
		if err != nil {
			logger.WithField("Error", err).Error("Failed to parse Session End for related session lookup")
			return nil, twirp.InternalError("Failed to parse Session End for related session lookup")
		}
	}

	logger.Debugf("Attempting LowRes query for %v", slChannel)
	resp, err := c.streamlog.GetLowres(slChannel, sessionStart, sessionEnd)
	if err != nil {
		logger.WithField("Error", err).Error("Failed to retrieve related session data from streamlog")
		return nil, twirp.InternalError("Failed to retrieve related session data")
	}

	if resp == nil {
		return nil, twirp.NotFoundError("No session data found")
	}

	// We're using a map while we populate the data to make it easier to reference the sessions by ID
	sessions := createSessionInfoList(resp.Records, logger)

	// Now that we've mapped the reponse from streamlog into sensible objects, add them to the list
	for _, relatedSession := range sessions {
		if relatedSession.Start == nil {
			// This can happen when the stream started before [start] but ended after, so there are some events for it
			// In this scenario, since is started before [start] we're just not going to return it
			continue
		}

		// We obviously don't want the main session in this list
		if relatedSession.SessionId == session.SessionId {
			continue
		}

		// Workaround for https://jira.twitch.com/browse/ING-6113
		goStart, err := ptypes.Timestamp(relatedSession.Start)
		if err == nil && time.Now().Sub(goStart) > time.Hour*48 {
			goEnd := goStart.Add(time.Hour * 48)
			pEnd, err := ptypes.TimestampProto(goEnd)
			if err == nil {
				relatedSession.End = pEnd
			}
		}

		if relatedSession.BroadcastFormat != "live" {
			session.RelatedSessions = append(session.RelatedSessions, relatedSession)
		}
	}

	// Sort the related sessions by streamup time asc
	sort.Slice(session.RelatedSessions, func(i, j int) bool {
		return session.RelatedSessions[i].Start.Seconds < session.RelatedSessions[j].Start.Seconds
	})

	// Hydrate info from mmdb
	if session.RtmpSession != nil && session.RtmpSession.ClientIp != "" {
		ipInfo, err := c.mmdb.GetIPInfo(session.RtmpSession.ClientIp)
		if err != nil {
			logger.WithError(err).Warn("Could not hydrate IP info from mmdb")
		}

		session.RtmpSession.IpInfo = ipInfo
	}

	return &csa.GetSessionDataResponse{
		Session: session,
	}, nil
}

func formatStreamlogChannel(customerID, region, contentID string) string {
	if customerID == "twitch" {
		return contentID
	}

	return fmt.Sprintf("aws.ivs.%s.%s.channel.%s", region, customerID, contentID)
}
