package liveproxyapi

import (
	"context"
	"fmt"
	"net/http"
	"strings"

	"code.justin.tv/video/liveproxyapi/rpc/liveproxyapi"
	emptypb "github.com/golang/protobuf/ptypes/empty"
	"github.com/sirupsen/logrus"
)

// Client defines the functions available on the LiveProxyAPI Client
type Client interface {
	GetProxyStatus(ctx context.Context) ([]*ProxyStatus, error)
}

type client struct {
	lpa    liveproxyapi.LiveProxyAPI
	logger logrus.FieldLogger
}

// New returns a new LiveProxyAPI Client
func New(liveProxyAPIURL string, logger logrus.FieldLogger) (Client, error) {
	splitURL := strings.Split(liveProxyAPIURL, ".")
	if len(splitURL) < 2 {
		return nil, fmt.Errorf("liveproxyapi URL doesn't seem to have a region: %s", liveProxyAPIURL)
	}

	signerClient, err := getSignerClient(http.DefaultClient, splitURL[0])

	if err != nil {
		return nil, err
	}

	return &client{
		logger: logger,
		lpa:    liveproxyapi.NewLiveProxyAPIProtobufClient(fmt.Sprintf("https://%s", liveProxyAPIURL), signerClient),
	}, nil
}

func (c *client) GetProxyStatus(ctx context.Context) ([]*ProxyStatus, error) {
	popHealthList, err := c.lpa.GetPOPHealthList(ctx, &emptypb.Empty{})

	if err != nil {
		return nil, err
	}

	ret := make([]*ProxyStatus, 0, len(popHealthList.Pops))

	for _, pop := range popHealthList.Pops {
		ret = append(ret, &ProxyStatus{
			ID:        pop.Id,
			Health:    pop.Health,
			DNS:       pop.Dns,
			Redirects: pop.Redirects,
			Name:      pop.Name,
			Type:      pop.Type,
			Lat:       pop.Lat,
			Long:      pop.Long,
		})
	}

	return ret, nil
}
