package omen

import (
	"bytes"
	"encoding/json"
	"io/ioutil"
	"net"
	"net/http"
	"net/url"
	"strings"
	"time"

	"code.justin.tv/neteng/omen/bin/common"
	"github.com/google/uuid"
	"github.com/sirupsen/logrus"
)

const omenPath = "/api/plyql/v1/query"

// Client defines the methods available in the omen client
type Client interface {
	GetBackbone() (map[string]int64, map[string]int64, map[string]int64, error)
	GetIngestQoS() (map[string]int64, map[string]float64, error)
	GetRTQoS() (map[string]int64, map[string]float64, error)
	GetEdge() (map[string]int64, map[string]int64, map[string]int64, error)
}

type client struct {
	omenHost    string
	druidBroker string
	omenApp     string
	logger      logrus.FieldLogger
}

// New returns a new omen client
func New(omenHost string, druidBroker string, omenApp string, logger logrus.FieldLogger) Client {
	return &client{
		omenHost:    omenHost,
		druidBroker: druidBroker,
		omenApp:     omenApp,
		logger:      logger,
	}
}

/*
	Returns a time filter for omen requests
*/
func getTimeFilter(minutes time.Duration) string {
	// We're shifting everything back by 1 minute so we don't have the issue of incomplete data in the last minute of the request
	timestamp := time.Now().Add(minutes).Add(time.Minute * -1)
	formattedTime := timestamp.Format("2006-01-02T15:04:05Z")
	nowFormatted := time.Now().Add(time.Minute * -1).Format("2006-01-02T15:04:05Z")
	return "__time > '" + formattedTime + "' AND __time <= '" + nowFormatted + "'"
}

func (c *client) getPopFromHostname(hostname string) (pop string) {
	return strings.Split(hostname, ".")[1]
}

/*
	Make a request to omen and return the response. If the
	result is cached, it will be returned from the cache.
*/
func (c *client) makeOmenRequest(request common.OmenRequest) ([]byte, error) {
	request.Uid = uuid.New().String()

	jsonValue, err := json.Marshal(request)
	if err != nil {
		return nil, err
	}

	// Make the reques to omen
	resp, err := http.Post(c.omenHost, "application/json", bytes.NewBuffer(jsonValue))
	// Print out any dns errors
	if err, ok := err.(*url.Error); ok {
		if err, ok := err.Err.(*net.OpError); ok {
			if _, ok := err.Err.(*net.DNSError); ok {
				c.logger.WithError(err).Warn("Failed to make request to omen")
			}
		}
	}

	if err != nil {
		return nil, err
	}

	defer resp.Body.Close()
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}

	return body, err
}
