package omen

import (
	"encoding/json"
	"strconv"
	"time"

	"code.justin.tv/neteng/omen/bin/common"
)

func (c *client) GetBackbone() (map[string]int64, map[string]int64, map[string]int64, error) {
	timeFilter := getTimeFilter(-1 * time.Minute)
	request := common.OmenRequest{
		Query:      "SELECT max(bpsout) as bpsout, max(bpsin) as bpsin, hostname, description, capacity FROM counter_stats WHERE description REGEXP '^COLO:.*:' AND " + timeFilter,
		Host:       c.druidBroker,
		Alias:      "{cluster}",
		Interval:   "1m",
		Downsample: false,
		App:        c.omenApp,
	}

	result, err := c.makeOmenRequest(request)
	if err != nil {
		return nil, nil, nil, err
	}

	var response OmenNetworkInterfaceResponse
	err = json.Unmarshal(result, &response)
	bbInMap := map[string]int64{}
	bbMaxMap := map[string]int64{}
	bbOutMap := map[string]int64{}

	for _, dataPoint := range response {
		pop := c.getPopFromHostname(dataPoint.Hostname)

		ifSpeed, err := strconv.ParseInt(dataPoint.IfSpeed, 10, 64)

		if err != nil {
			c.logger.Errorf("Failed to parse link capacity: %v", dataPoint.IfSpeed)
		}

		if _, ok := bbMaxMap[pop]; ok {
			bbMaxMap[pop] += ifSpeed
		} else {
			bbMaxMap[pop] = ifSpeed
		}

		if _, ok := bbInMap[pop]; ok {
			bbInMap[pop] += dataPoint.BpsIn
		} else {
			bbInMap[pop] = dataPoint.BpsIn
		}

		if _, ok := bbOutMap[pop]; ok {
			bbOutMap[pop] += dataPoint.BpsOut
		} else {
			bbOutMap[pop] = dataPoint.BpsOut
		}
	}
	return bbInMap, bbOutMap, bbMaxMap, err
}

func (c *client) GetEdge() (map[string]int64, map[string]int64, map[string]int64, error) {
	timeFilter := getTimeFilter(-1 * time.Minute)
	request := common.OmenRequest{
		Query:      "SELECT max(bpsout) as bpsout, max(bpsin) as bpsin, hostname, description, capacity FROM counter_stats WHERE capacity != '0' and description REGEXP '^(PEER-(IX|PNI)|ISP):.*:' AND " + timeFilter,
		Host:       c.druidBroker,
		Alias:      "{cluster}",
		Interval:   "1m",
		Downsample: false,
		App:        c.omenApp,
	}

	result, err := c.makeOmenRequest(request)
	if err != nil {
		return nil, nil, nil, err
	}

	var response OmenNetworkInterfaceResponse
	err = json.Unmarshal(result, &response)
	bbInMap := map[string]int64{}
	bbMaxMap := map[string]int64{}
	bbOutMap := map[string]int64{}

	for _, dataPoint := range response {
		pop := c.getPopFromHostname(dataPoint.Hostname)

		ifSpeed, err := strconv.ParseInt(dataPoint.IfSpeed, 10, 64)

		if err != nil {
			c.logger.Errorf("Failed to parse link capacity: %v", dataPoint.IfSpeed)
		}

		if _, ok := bbMaxMap[pop]; ok {
			bbMaxMap[pop] += ifSpeed
		} else {
			bbMaxMap[pop] = ifSpeed
		}

		if _, ok := bbInMap[pop]; ok {
			bbInMap[pop] += dataPoint.BpsIn
		} else {
			bbInMap[pop] = dataPoint.BpsIn
		}

		if _, ok := bbOutMap[pop]; ok {
			bbOutMap[pop] += dataPoint.BpsOut
		} else {
			bbOutMap[pop] = dataPoint.BpsOut
		}
	}
	return bbInMap, bbOutMap, bbMaxMap, err
}

func (c *client) GetIngestQoS() (map[string]int64, map[string]float64, error) {
	timeFilter := getTimeFilter(-1 * time.Minute)
	request := common.OmenRequest{
		Query:      "SELECT cluster, count(*) AS CCB, sum(Starvations)/sum(StarvationSamples) AS 'PTS' FROM ingest_qos WHERE " + timeFilter + " GROUP BY cluster",
		Host:       c.druidBroker,
		Alias:      "{cluster}",
		Interval:   "1m",
		Downsample: false,
		App:        c.omenApp,
	}

	result, err := c.makeOmenRequest(request)
	if err != nil {
		return nil, nil, err
	}

	var response OmenIngestQoSResponse
	err = json.Unmarshal(result, &response)
	ccbMap := map[string]int64{}
	ptsMap := map[string]float64{}

	for _, data := range response {
		ccbMap[data.Cluster] = data.CCB
		ptsMap[data.Cluster] = data.PTS
	}

	return ccbMap, ptsMap, err
}

/*
	Returns the be/mw and mw per PoP
*/
func (c *client) GetRTQoS() (map[string]int64, map[string]float64, error) {
	// Make the request to omen
	timeFilter := getTimeFilter(-1 * time.Minute)
	request := common.OmenRequest{
		Query:      "SELECT cluster, sum(MinutesWatched) AS MW, (sum(BufferEmpties)/sum(MinutesWatched)) as 'BEMW' FROM realtime_qos WHERE NOT cluster CONTAINS 'twitch' AND " + timeFilter + " GROUP BY cluster",
		Host:       c.druidBroker,
		Alias:      "{cluster}",
		Interval:   "1m",
		Downsample: false,
		App:        c.omenApp,
	}

	result, err := c.makeOmenRequest(request)
	if err != nil {
		return nil, nil, err
	}

	var response OmenRTQoSResponse
	err = json.Unmarshal(result, &response)
	// Parse the response into per pop map
	mwMap := map[string]int64{}
	bemwMap := map[string]float64{}

	for _, data := range response {
		mwMap[data.Cluster] = data.MW
		bemwMap[data.Cluster] = data.BEMW
	}

	return mwMap, bemwMap, err
}
