package svc

import (
	"context"
	"fmt"
	"sort"
	"strings"
	"sync"
	"time"

	"code.justin.tv/event-engineering/carrot-system-health/pkg/brood"
	"code.justin.tv/event-engineering/carrot-system-health/pkg/liveproxyapi"
	"code.justin.tv/event-engineering/carrot-system-health/pkg/omen"
	csh "code.justin.tv/event-engineering/carrot-system-health/pkg/rpc"
	rpc "code.justin.tv/event-engineering/carrot-system-health/pkg/rpc"
	"github.com/sirupsen/logrus"
)

// Client defines the functions that will be available in this service, in this case it's pretty much a straight implementation of the twirp service
type Client interface {
	GetPoPHealth(context context.Context, request *csh.GetPoPHealthRequest) (*csh.GetPoPHealthResponse, error)
}

type client struct {
	omenClient         omen.Client
	broodClient        brood.Client
	liveProxyAPIClient liveproxyapi.Client
	logger             logrus.FieldLogger
	originDCs          []string
}

// New returns a new Carrot System Health client
func New(omenHost, druidBroker, omenAppName, broodAPIURL, liveproxyAPIURL string, originDCs []string, logger logrus.FieldLogger) (Client, error) {
	bc, err := brood.New(broodAPIURL)
	if err != nil {
		return nil, err
	}

	lpa, err := liveproxyapi.New(liveproxyAPIURL, logger)

	if err != nil {
		return nil, err
	}

	return &client{
		omenClient:         omen.New(omenHost, druidBroker, omenAppName, logger),
		liveProxyAPIClient: lpa,
		broodClient:        bc,
		originDCs:          originDCs,
		logger:             logger,
	}, nil
}

func (c *client) GetPoPHealth(ctx context.Context, request *csh.GetPoPHealthRequest) (*csh.GetPoPHealthResponse, error) {
	var wg sync.WaitGroup
	errCh := make(chan error)
	popHealth := &csh.GetPoPHealthResponse{}

	ctx, cancel := context.WithTimeout(ctx, 10*time.Second)
	defer cancel()

	errors := make([]error, 0)

	go func() {
		for {
			err, ok := <-errCh
			if !ok {
				return
			}

			errors = append(errors, err)
		}
	}()

	// RTQoS
	wg.Add(1)
	go func(ictx context.Context, ph *rpc.GetPoPHealthResponse) {
		defer wg.Done()

		mw, bemw, err := c.omenClient.GetRTQoS()
		if err != nil {
			errCh <- fmt.Errorf("RTQoS: %v", err)
			return
		}

		for pop, val := range mw {
			poph := getOrSetPop(ph, pop)
			if poph != nil {
				poph.MinutesWatched = val
			}
		}

		for pop, val := range bemw {
			poph := getOrSetPop(ph, pop)
			if poph != nil {
				poph.BufferEmptiesPerMw = val
			}
		}

	}(ctx, popHealth)

	// IngestQoS
	wg.Add(1)
	go func(ictx context.Context, ph *rpc.GetPoPHealthResponse) {
		defer wg.Done()

		ccb, pts, err := c.omenClient.GetIngestQoS()
		if err != nil {
			errCh <- fmt.Errorf("IngestQoS: %v", err)
			return
		}

		for pop, val := range pts {
			poph := getOrSetPop(ph, pop)
			if poph != nil {
				poph.PercentTimeStarving = val
			}
		}

		for pop, val := range ccb {
			poph := getOrSetPop(ph, pop)
			if poph != nil {
				poph.ConcurrentBroadcasters = val
			}
		}
	}(ctx, popHealth)

	// Backbone
	wg.Add(1)
	go func(ictx context.Context, ph *rpc.GetPoPHealthResponse) {
		defer wg.Done()

		bbIn, bbOut, bbMax, err := c.omenClient.GetBackbone()
		if err != nil {
			errCh <- fmt.Errorf("Backbone: %v", err)
			return
		}

		for pop, val := range bbIn {
			poph := getOrSetPop(ph, pop)
			if poph != nil {
				poph.BackboneIn = val
			}
		}

		for pop, val := range bbOut {
			poph := getOrSetPop(ph, pop)
			if poph != nil {
				poph.BackboneOut = val
			}
		}

		for pop, val := range bbMax {
			poph := getOrSetPop(ph, pop)
			if poph != nil {
				poph.BackboneMax = val
			}
		}
	}(ctx, popHealth)

	// Edge
	wg.Add(1)
	go func(ictx context.Context, ph *rpc.GetPoPHealthResponse) {
		defer wg.Done()

		bbIn, bbOut, bbMax, err := c.omenClient.GetEdge()
		if err != nil {
			errCh <- fmt.Errorf("Edge: %v", err)
			return
		}

		for pop, val := range bbIn {
			poph := getOrSetPop(ph, pop)
			if poph != nil {
				poph.EdgeIn = val
			}
		}

		for pop, val := range bbOut {
			poph := getOrSetPop(ph, pop)
			if poph != nil {
				poph.EdgeOut = val
			}
		}

		for pop, val := range bbMax {
			poph := getOrSetPop(ph, pop)
			if poph != nil {
				poph.EdgeMax = val
			}
		}
	}(ctx, popHealth)

	wg.Wait()

	// Bit of a hack for now, but we're waiting until all the other requests are done before populating the data from brood/liveproxyapi
	// we can do this more efficiently at some point

	// Brood Clusters
	wg.Add(1)
	go func(ictx context.Context, ph *rpc.GetPoPHealthResponse) {
		defer wg.Done()

		clusters, err := c.broodClient.GetClusterList(ictx)
		if err != nil {
			errCh <- fmt.Errorf("Clusters: %v", err)
			return
		}

		for pop, cluster := range clusters {
			// Results from brood shouldn't create new entries in the list
			poph := getPop(ph, pop)
			if poph != nil {

				poph.IsServingEdge = cluster.IsServingEdge
				poph.IsServingWeaver = cluster.IsServingWeaver
				poph.NetworkEnvironment = cluster.NetworkEnvironment
			}
		}

	}(ctx, popHealth)

	// Liveproxyapi
	wg.Add(1)
	go func(ictx context.Context, ph *rpc.GetPoPHealthResponse) {
		defer wg.Done()

		ingestPops, err := c.liveProxyAPIClient.GetProxyStatus(ictx)
		if err != nil {
			errCh <- fmt.Errorf("ProxyStatus: %v", err)
			return
		}

		for _, ip := range ingestPops {
			if ip.Type == "private" {
				continue
			}

			poph := getPopFromIngestID(ph, ip.ID)

			if poph != nil {
				poph.IsServingIngest = ip.ID == ip.Redirects
				poph.IngestHealthStatus = ip.Health
				poph.Lat = ip.Lat
				poph.Long = ip.Long
			}
		}

	}(ctx, popHealth)

	wg.Wait()
	close(errCh)

	for _, pop := range popHealth.Pops {
		pop.IsOriginDc = c.isOrigin(pop.Id)
	}

	if popHealth.Pops != nil {
		sort.Slice(popHealth.Pops, func(i, j int) bool {
			return popHealth.Pops[i].MinutesWatched > popHealth.Pops[j].MinutesWatched
		})
	}

	if len(errors) > 0 {
		return popHealth, fmt.Errorf("Error retrieving PoP health %v", errors)
	}

	return popHealth, nil
}

func (c *client) isOrigin(pop string) bool {
	for _, origin := range c.originDCs {
		if strings.EqualFold(origin, pop) {
			return true
		}
	}

	return false
}
