import * as cdk from '@aws-cdk/core';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as route53 from '@aws-cdk/aws-route53';

export interface Delegation {
    name: string
    hostedZoneName: string
    nameServers: string[] | undefined
}

export interface CarrotDnsProps {
    cdkProps: cdk.StackProps;
    zoneName: string
    delegations: Delegation[]
}

export class CarrotDnsStack extends cdk.Stack {
    RTMPCaptureZone: route53.PublicHostedZone

    constructor(scope: cdk.Construct, id: string, vpc: ec2.IVpc, props: CarrotDnsProps) {
        super(scope, id, props.cdkProps);

        this.RTMPCaptureZone = new route53.PublicHostedZone(this, 'RTMPCaptureZone', {
            zoneName: props.zoneName
        });

        props.delegations.forEach((d: Delegation) => {
            if (d.nameServers !== undefined) {
                new route53.NsRecord(this, d.name, {
                    zone: this.RTMPCaptureZone,
                    recordName: d.hostedZoneName,
                    values: d.nameServers,
                })
            }
        })
    }
}
