import elasticache = require('@aws-cdk/aws-elasticache')
import ec2 = require('@aws-cdk/aws-ec2');
import cdk = require('@aws-cdk/core');

export class CarrotCacheCluster extends cdk.Construct {
  RedisCluster: elasticache.CfnReplicationGroup

  constructor(scope: cdk.Construct, id: string, vpc: ec2.IVpc) {
    super(scope, id);

    var privateSubnets: string[] = [];

    for (let subnet of vpc.privateSubnets) {
      privateSubnets.push(subnet.subnetId)
    }

    let subnetGroup = new elasticache.CfnSubnetGroup(this, 'CarrotRedisSubnetGroup', {
      cacheSubnetGroupName: 'carrot-cache-subnet-group',
      description: 'Subnet group for the elasticache cluster',
      subnetIds: privateSubnets,
    })

    let securityGroupIds: string[] = [];

    let securityGroup = new ec2.SecurityGroup(this, 'CarrotRedisSecurityGroup', {
      allowAllOutbound: true,
      description: 'Allow cache cluster access from VPC',
      vpc: vpc,
    })

    securityGroup.addIngressRule(ec2.Peer.ipv4(vpc.vpcCidrBlock), ec2.Port.tcp(6379))

    securityGroupIds.push(securityGroup.securityGroupId)

    this.RedisCluster = new elasticache.CfnReplicationGroup(this, 'CarrotRedisCluster', {
      engine: 'redis',
      replicationGroupDescription: 'Redis cluster for carrot',
      cacheNodeType: 'cache.t3.small',
      automaticFailoverEnabled: true,
      replicasPerNodeGroup: 1,
      numNodeGroups: 2,
      atRestEncryptionEnabled: true,
      transitEncryptionEnabled: true, // Auth will be added OOB because I don't want to put keys in CDK
      port: 6379,
      cacheSubnetGroupName: subnetGroup.cacheSubnetGroupName,
      securityGroupIds: securityGroupIds,
    })

  }
}
