import * as cdk from '@aws-cdk/core';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as route53resolver from '@aws-cdk/aws-route53resolver';
import { CarrotCacheCluster } from './carrot-memory-cache/carrot-memory-cache'

export interface CarrotVpcProps {
  cdkProps: cdk.StackProps;
  cidr: string
}

export class CarrotVpcStack extends cdk.Stack {
  Vpc: ec2.Vpc;
  CacheCluster: CarrotCacheCluster;

  constructor(scope: cdk.Construct, id: string, props: CarrotVpcProps) {
    super(scope, id, props.cdkProps);

    this.Vpc = new ec2.Vpc(this, 'CarrotVpc', {
      maxAzs: 3,
      cidr: props.cidr,
    })

    const resolverSecurityGroup = new ec2.SecurityGroup(this, 'CarrotInternalResolverSG', {
      vpc: this.Vpc,
      description: 'Allow internal resolver to query internal DNS',
      allowAllOutbound: false,
    })

    resolverSecurityGroup.addEgressRule(ec2.Peer.anyIpv4(), ec2.Port.udp(53), 'Allow DNS resolution')

    var ipAddresses: any[] = [];

    for (let subnet of this.Vpc.privateSubnets) {
      ipAddresses.push({
        subnetId: subnet.subnetId,
      })
    }

    const resolverEndpoint = new route53resolver.CfnResolverEndpoint(this, 'CarrotInternalResolverEndpoint', {
      direction: 'OUTBOUND',
      ipAddresses: ipAddresses,
      securityGroupIds: [resolverSecurityGroup.securityGroupId],
    })

    const resolverRule = new route53resolver.CfnResolverRule(this, 'CarrotInternalResolverRule', {
      name: 'internal forwarder',
      domainName: 'justin.tv',
      resolverEndpointId: resolverEndpoint.attrResolverEndpointId,
      ruleType: 'FORWARD',
      targetIps: [
        {
          ip: '10.254.0.3',
        },
        {
          ip: '10.254.0.8',
        }
      ]
    })

    new route53resolver.CfnResolverRuleAssociation(this, 'CarrotInternalResolverAssociation', {
      vpcId: this.Vpc.vpcId,
      resolverRuleId: resolverRule.attrResolverRuleId,
    })

    // Never actually used this
    // new CarrotCacheCluster(this, 'CarrotCacheCluster', this.Vpc)
  }
}
