package main

import (
	"context"
	"time"

	"strconv"
	"strings"

	"github.com/Sirupsen/logrus"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-lambda-go/lambda"
	"github.com/beevik/ntp"
)

type syncState struct {
	lastSync    time.Time
	clockOffset time.Duration
}

var (
	logger = logrus.New()
	state  = make(map[string]syncState)
)

func main() {
	logger.Level = logrus.DebugLevel
	lambda.Start(Handle)
}

func Handle(ctx context.Context, request events.APIGatewayProxyRequest) (events.APIGatewayProxyResponse, error) {
	var timestampstr string
	var ok bool
	var resp = events.APIGatewayProxyResponse{
		Headers: map[string]string{
			"Content-Type":                "text/plain",
			"Access-Control-Allow-Origin": "*",
		},
	}

	if timestampstr, ok = request.QueryStringParameters["t"]; !ok {
		logger.Warn("No timestamp passed")
		resp.StatusCode = 400
		resp.Body = "No Timestamp passed"
		return resp, nil
	}

	timestamp, err := strconv.ParseInt(timestampstr, 10, 64)
	if err != nil {
		logger.Warn("Failed to parse timestamp")
		resp.StatusCode = 400
		resp.Body = "Failed to parse timestamp"
		return resp, nil
	}

	var ntpServer = ""
	if ns, ok := request.QueryStringParameters["s"]; ok {
		ntpServer = ns
	}

	// If we've been given a specific NTP server then use it, othwerise just use the system time
	var theTime time.Time
	if ntpServer != "" {
		syncServer := false
		if ss, ok := state[strings.ToLower(ntpServer)]; ok {
			// Use the last offset we worked out from NTP
			if ss.lastSync.Add(10 * time.Minute).Before(time.Now()) {
				syncServer = true
			} else {
				theTime = time.Now().Add(ss.clockOffset)
			}
		} else {
			syncServer = true
		}

		if syncServer {
			response, err := ntp.Query(ntpServer)
			sysTime := time.Now()
			theTime = sysTime.Add(response.ClockOffset)

			state[strings.ToLower(ntpServer)] = syncState{
				lastSync:    sysTime,
				clockOffset: response.ClockOffset,
			}

			logger.Debugf("Got time %v from server %v", theTime, ntpServer)
			if err != nil {
				resp.StatusCode = 400
				resp.Body = "Failed to connect to NTP server"
				logger.Warn("Failed to connect to NTP server ", err)
				return resp, nil
			}
		}
	} else {
		theTime = time.Now()
	}

	// Convert to unix milliseconds
	now := theTime.UTC().UnixNano() / int64(time.Millisecond)

	// Calculate offset
	offset := now - timestamp

	// return data in format offset:original_timestamp
	resp.Body = strconv.FormatInt(offset, 10) + ":" + strconv.FormatInt(timestamp, 10)
	resp.StatusCode = 200

	return resp, nil
}
