package destination

import (
	"encoding/json"
	"fmt"
	"log"
	"net/rpc"

	"code.justin.tv/event-engineering/covfefe/pkg/server/api"
	"github.com/urfave/cli"
)

// Configure the destonation command
func Configure(client *rpc.Client, commands *[]cli.Command) {
	destinationCmd := cli.Command{
		Name:    "dest",
		Aliases: []string{"d"},
		Usage:   "options for destinations",
		Subcommands: []cli.Command{
			{
				Name:  "add",
				Usage: "add a new destination",
				Flags: []cli.Flag{
					cli.StringFlag{
						Name:  "name",
						Value: "",
						Usage: "optional destination name",
					},
				},
				Subcommands: []cli.Command{
					{
						Name:  "srtmp",
						Usage: "add a serve RTMP destination",
						Flags: []cli.Flag{
							cli.StringFlag{
								Name:  "app",
								Value: "app",
								Usage: "app name to listen on",
							},
							cli.IntFlag{
								Name:  "port",
								Value: 1935,
								Usage: "port to listen on",
							},
						},
						Action: func(c *cli.Context) error {
							handleAddSRTMP(client, c.String("app"), c.Int("port"))
							return nil
						},
					},
					{
						Name:  "prtmp",
						Usage: "add a push RTMP destination",
						Action: func(c *cli.Context) error {
							if c.NArg() == 0 {
								fmt.Println("No RTMP url passed")
								return nil
							}

							handleAddPRTMP(client, c.Args().Get(0))
							return nil
						},
					},
					{
						Name:  "fs",
						Usage: "add a file sink RTMP destination",
						Action: func(c *cli.Context) error {
							if c.NArg() == 0 {
								fmt.Println("No file path passed")
								return nil
							}

							handleAddFS(client, c.Args().Get(0))
							return nil
						},
					},
				},
			},
			{
				Name:  "list",
				Usage: "list destinations",
				Action: func(c *cli.Context) error {
					handleListDestinations(client)
					return nil
				},
			},
			{
				Name:  "delete",
				Usage: "delete a single, or multiple destinations",
				Action: func(c *cli.Context) error {
					if c.NArg() == 0 {
						fmt.Println("No destination id passed")
						return nil
					}
					for _, arg := range c.Args() {
						handleRemoveDestination(client, arg)
					}

					return nil
				},
			},
		},
	}

	*commands = append(*commands, destinationCmd)
}

func handleAddSRTMP(client *rpc.Client, app string, port int) {
	cmdArgs := &api.AddDestSRTMPArgs{Port: port, AppName: app}
	var reply api.AddDestSRTMPResponse

	err := client.Call("Covfefe.AddDestSRTMP", cmdArgs, &reply)
	if err != nil {
		log.Println("AddDestSRTMP error:", err)
	} else {
		body, _ := json.Marshal(reply)

		fmt.Printf("AddDestSRTMP: %v\n", string(body))
	}
}

func handleAddPRTMP(client *rpc.Client, url string) {
	cmdArgs := &api.AddDestPRTMPArgs{URL: url}
	var reply api.AddDestPRTMPResponse

	err := client.Call("Covfefe.AddDestPRTMP", cmdArgs, &reply)
	if err != nil {
		log.Println("AddDestPRTMP error:", err)
	} else {
		body, _ := json.Marshal(reply)

		fmt.Printf("AddDestPRTMP: %v\n", string(body))
	}
}

func handleAddFS(client *rpc.Client, filePath string) {
	cmdArgs := &api.AddDestFSArgs{FilePath: filePath}
	var reply api.AddDestFSResponse

	err := client.Call("Covfefe.AddDestFS", cmdArgs, &reply)
	if err != nil {
		log.Println("AddDestFS error:", err)
	} else {
		body, _ := json.Marshal(reply)

		fmt.Printf("AddDestFS: %v\n", string(body))
	}
}

func handleRemoveDestination(client *rpc.Client, ID string) {
	cmdArgs := &api.RemoveDestinationArgs{ID: ID}
	var reply api.RemoveDestinationResponse

	err := client.Call("Covfefe.RemoveDestination", cmdArgs, &reply)
	if err != nil {
		log.Println("RemoveDestination error:", err)
	} else {
		body, _ := json.Marshal(reply)

		fmt.Printf("RemoveDestination: %v\n", string(body))
	}
}

func handleListDestinations(client *rpc.Client) {
	cmdArgs := &api.ListDestinationsArgs{}
	var reply api.ListDestinationsResponse

	err := client.Call("Covfefe.ListDestinations", cmdArgs, &reply)
	if err != nil {
		log.Println("ListDestinations error:", err)
	} else {
		body, _ := json.Marshal(reply)

		fmt.Printf("%v\n", string(body))
	}
}
