package route

import (
	"encoding/json"
	"fmt"
	"log"
	"net/rpc"

	"code.justin.tv/event-engineering/covfefe/pkg/server/api"
	"github.com/urfave/cli"
)

// Configure the destonation command
func Configure(client *rpc.Client, commands *[]cli.Command) {
	routeCmd := cli.Command{
		Name:    "route",
		Aliases: []string{"r"},
		Usage:   "options for routes",
		Subcommands: []cli.Command{
			{
				Name:  "add",
				Usage: "add a new route for a specified source",
				Flags: []cli.Flag{
					cli.StringFlag{
						Name:  "name",
						Value: "",
						Usage: "optional route name",
					},
				},
				Action: func(c *cli.Context) error {
					if c.NArg() == 0 {
						fmt.Println("No source id passed")
						return nil
					}

					handleAddRoute(client, c.Args().Get(0))

					return nil
				},
			},
			{
				Name:  "link",
				Usage: "link a destination to the specified source",
				Action: func(c *cli.Context) error {
					if c.NArg() == 0 {
						fmt.Println("No source id passed")
						return nil
					}

					if c.NArg() == 1 {
						fmt.Println("No destination id passed")
						return nil
					}

					handleLinkRoute(client, c.Args().Get(0), c.Args().Get(1))

					return nil
				},
			},
			{
				Name:  "unlink",
				Usage: "unlink a destination from the specified source",
				Action: func(c *cli.Context) error {
					if c.NArg() == 0 {
						fmt.Println("No source id passed")
						return nil
					}

					if c.NArg() == 1 {
						fmt.Println("No destination id passed")
						return nil
					}

					handleUnlinkRoute(client, c.Args().Get(0), c.Args().Get(1))

					return nil
				},
			},
			{
				Name:  "activate",
				Usage: "activate a destination for the specified source",
				Action: func(c *cli.Context) error {
					if c.NArg() == 0 {
						fmt.Println("No source id passed")
						return nil
					}

					if c.NArg() == 1 {
						fmt.Println("No destination id passed")
						return nil
					}

					handleActivateRoute(client, c.Args().Get(0), c.Args().Get(1))

					return nil
				},
			},
			{
				Name:  "deactivate",
				Usage: "deactivate a destination for the specified source",
				Action: func(c *cli.Context) error {
					if c.NArg() == 0 {
						fmt.Println("No source id passed")
						return nil
					}

					if c.NArg() == 1 {
						fmt.Println("No destination id passed")
						return nil
					}

					handleDeactivateRoute(client, c.Args().Get(0), c.Args().Get(1))

					return nil
				},
			},
			{
				Name:  "list",
				Usage: "list routes",
				Action: func(c *cli.Context) error {
					handleListRoutes(client)
					return nil
				},
			},
			{
				Name:  "delete",
				Usage: "delete a single, or multiple routes",
				Action: func(c *cli.Context) error {
					if c.NArg() == 0 {
						fmt.Println("No source id passed")
						return nil
					}
					for _, arg := range c.Args() {
						handleRemoveRoute(client, arg)
					}

					return nil
				},
			},
		},
	}

	*commands = append(*commands, routeCmd)
}

func handleAddRoute(client *rpc.Client, sourceID string) {
	cmdArgs := &api.AddRouteArgs{SourceID: sourceID}
	var reply api.AddRouteResponse

	err := client.Call("Covfefe.AddRoute", cmdArgs, &reply)
	if err != nil {
		log.Println("AddRoute error:", err)
	} else {
		body, _ := json.Marshal(reply)

		fmt.Printf("AddRoute: %v\n", string(body))
	}
}

func handleLinkRoute(client *rpc.Client, sourceID string, destinationID string) {
	cmdArgs := &api.LinkRouteArgs{SourceID: sourceID, DestinationID: destinationID}
	var reply api.LinkRouteResponse

	err := client.Call("Covfefe.LinkRoute", cmdArgs, &reply)
	if err != nil {
		log.Println("LinkRoute error:", err)
	} else {
		body, _ := json.Marshal(reply)

		fmt.Printf("LinkRoute: %v\n", string(body))
	}
}

func handleUnlinkRoute(client *rpc.Client, sourceID string, destinationID string) {
	cmdArgs := &api.UnlinkRouteArgs{SourceID: sourceID, DestinationID: destinationID}
	var reply api.UnlinkRouteResponse

	err := client.Call("Covfefe.UnlinkRoute", cmdArgs, &reply)
	if err != nil {
		log.Println("UnlinkRoute error:", err)
	} else {
		body, _ := json.Marshal(reply)

		fmt.Printf("UnlinkRoute: %v\n", string(body))
	}
}

func handleActivateRoute(client *rpc.Client, sourceID string, destinationID string) {
	cmdArgs := &api.ActivateRouteArgs{SourceID: sourceID, DestinationID: destinationID}
	var reply api.ActivateRouteResponse

	err := client.Call("Covfefe.ActivateRoute", cmdArgs, &reply)
	if err != nil {
		log.Println("ActivateRoute error:", err)
	} else {
		body, _ := json.Marshal(reply)

		fmt.Printf("ActivateRoute: %v\n", string(body))
	}
}

func handleDeactivateRoute(client *rpc.Client, sourceID string, destinationID string) {
	cmdArgs := &api.DeactivateRouteArgs{SourceID: sourceID, DestinationID: destinationID}
	var reply api.DeactivateRouteResponse

	err := client.Call("Covfefe.DeactivateRoute", cmdArgs, &reply)
	if err != nil {
		log.Println("DeactivateRoute error:", err)
	} else {
		body, _ := json.Marshal(reply)

		fmt.Printf("DeactivateRoute: %v\n", string(body))
	}
}

func handleRemoveRoute(client *rpc.Client, sourceID string) {
	cmdArgs := &api.RemoveRouteArgs{SourceID: sourceID}
	var reply api.RemoveRouteResponse

	err := client.Call("Covfefe.RemoveRoute", cmdArgs, &reply)
	if err != nil {
		log.Println("RemoveRoute error:", err)
	} else {
		body, _ := json.Marshal(reply)

		fmt.Printf("RemoveRoute: %v\n", string(body))
	}
}

func handleListRoutes(client *rpc.Client) {
	cmdArgs := &api.ListRoutesArgs{}
	var reply api.ListRoutesResponse

	err := client.Call("Covfefe.ListRoutes", cmdArgs, &reply)
	if err != nil {
		log.Println("ListRoutes error:", err)
	} else {
		body, _ := json.Marshal(reply)

		fmt.Printf("%v\n", string(body))
	}
}
