package source

import (
	"encoding/json"
	"fmt"
	"log"
	"net/rpc"

	"code.justin.tv/event-engineering/covfefe/pkg/server/api"
	"github.com/urfave/cli"
)

// Configure the source command
func Configure(client *rpc.Client, commands *[]cli.Command) {
	sourceCmd := cli.Command{
		Name:    "source",
		Aliases: []string{"s"},
		Usage:   "options for sources",
		Subcommands: []cli.Command{
			{
				Name:  "add",
				Usage: "add a new source",
				Flags: []cli.Flag{
					cli.StringFlag{
						Name:  "name",
						Value: "",
						Usage: "optional source name",
					},
				},
				Subcommands: []cli.Command{
					{
						Name:  "lrtmp",
						Usage: "add a listen RTMP source",
						Flags: []cli.Flag{
							cli.StringFlag{
								Name:  "app",
								Value: "app",
								Usage: "app name to listen on",
							},
							cli.IntFlag{
								Name:  "port",
								Value: 1935,
								Usage: "port to listen on",
							},
						},
						Action: func(c *cli.Context) error {
							handleAddSourceLRTMP(client, c.String("app"), c.Int("port"))
							return nil
						},
					},
				},
			},
			{
				Name:  "list",
				Usage: "list sources",
				Action: func(c *cli.Context) error {
					handleListSources(client)
					return nil
				},
			},
			{
				Name:  "delete",
				Usage: "delete a single, or multiple sources",
				Action: func(c *cli.Context) error {
					if c.NArg() == 0 {
						fmt.Println("No source id passed")
						return nil
					}

					for _, arg := range c.Args() {
						handleRemoveSource(client, arg)
					}

					return nil
				},
			},
		},
	}

	*commands = append(*commands, sourceCmd)
}

func handleAddSourceLRTMP(client *rpc.Client, app string, port int) {
	cmdArgs := &api.AddSourceRTMPLArgs{Port: port, AppName: app}
	var reply api.AddSourceRTMPLResponse

	err := client.Call("Covfefe.AddSourceRTMPL", cmdArgs, &reply)
	if err != nil {
		log.Println("AddSourceRTMPL error:", err)
	} else {
		body, _ := json.Marshal(reply)

		fmt.Printf("AddSourceRTMPL: %v\n", string(body))
	}
}

func handleRemoveSource(client *rpc.Client, ID string) {
	cmdArgs := &api.RemoveSourceArgs{ID: ID}
	var reply api.RemoveSourceResponse

	err := client.Call("Covfefe.RemoveSource", cmdArgs, &reply)
	if err != nil {
		log.Println("RemoveSource error:", err)
	} else {
		body, _ := json.Marshal(reply)

		fmt.Printf("RemoveSource: %v\n", string(body))
	}
}

func handleListSources(client *rpc.Client) {
	cmdArgs := &api.ListSourcesArgs{}
	var reply api.ListSourcesResponse

	err := client.Call("Covfefe.ListSources", cmdArgs, &reply)
	if err != nil {
		log.Println("ListSource error:", err)
	} else {
		body, _ := json.Marshal(reply)

		fmt.Printf("%v\n", string(body))
	}
}
