package control

import (
	"encoding/json"
	"errors"
	"fmt"
	"log"
	"net"
	"strconv"

	"code.justin.tv/event-engineering/covfefe/pkg/server/api"
	"code.justin.tv/event-engineering/covfefe/pkg/server/db"
	"code.justin.tv/event-engineering/covfefe/pkg/server/router"
	"code.justin.tv/event-engineering/covfefe/pkg/server/rtmp"
	"code.justin.tv/event-engineering/covfefe/pkg/server/state"

	"code.justin.tv/event-engineering/covfefe/pkg/server/rpc"
)

var database *db.Covfefe
var currentState state.State

// SetupController set up control handlers for the server
func SetupController() {
	database = db.Init("covfefe.db")

	// Load the state from the db
	loadState()

	// Execute the state to set up anything that was stored in the db
	err := Execute(&currentState)
	if err != nil {
		// If the inital state that's loaded from the db can't be executed then something is very wrong so exit
		log.Fatalf("Could not execute stored state, this requires some manual intervention - usually deleting the db and starting again\n%v", err)
	}

	// Set up listeners so listen for control commands
	var listeners = generateListeners()
	for _, listener := range listeners {
		listener.OnAddSourceRTMPL = handleAddSourceRTMPL
		listener.OnRemoveSource = handleRemoveSource
		listener.OnListSources = handleListSources
		listener.OnAddDestSRTMP = handleAddDestSRTMP
		listener.OnAddDestPRTMP = handleAddDestPRTMP
		listener.OnAddDestFS = handleAddDestFS
		listener.OnListDestinations = handleListDestinations
		listener.OnRemoveDestination = handleRemoveDestination
		listener.OnAddRoute = handleAddRoute
		listener.OnLinkRoute = handleLinkRoute
		listener.OnUnlinkRoute = handleUnlinkRoute
		listener.OnActivateRoute = handleActivateRoute
		listener.OnDeactivateRoute = handleDeactivateRoute
		listener.OnRemoveRoute = handleRemoveRoute
		listener.OnListRoutes = handleListRoutes
	}
}

// HandleExit handle any cleaning up we want to do
func HandleExit() {
	fmt.Println("Cleaning up")

	// Close db
	database.Close()
}

// generateListeners Generate listeners for the api
func generateListeners() []*api.Listener {
	var listeners = make([]*api.Listener, 0)

	// TODO - work out how we pass config here
	var rpcListener = rpc.GenerateListener(1337)
	listeners = append(listeners, rpcListener)

	// If we decide to support other types of listener, e.g. REST API or websocket or carrier pigeon they would go here

	return listeners
}

func loadState() {
	// Read the state from the db
	stateData, err := database.ReadState()
	if err != nil {
		log.Fatal("Error reading state ", err)
	}

	if stateData == nil {
		return
	}

	// Load the state from the db into the state object
	err = json.Unmarshal(stateData, &currentState)

	if err != nil {
		log.Fatal("Error deserialising state ", err)
	}
}

func GenerateAPIListeners() {

}

func commitState() error {
	err := Execute(&currentState)
	if err != nil {
		loadState()
		return err
	}

	var data []byte
	data, err = json.Marshal(currentState)
	if err != nil {
		fmt.Println("Error serialising state ", err)
		loadState()
		return err
	}

	err = database.WriteState(data)
	if err != nil {
		fmt.Println("Error writing state ", err)
		loadState()
		return err
	}

	return nil
}

// Execute sync the state object to execute any staged changes
func Execute(s *state.State) error {
	fmt.Printf("State has %v sources, %v destinations and %v Routes\n", len(s.Sources), len(s.Destinations), len(s.Routes))

	// Add any sources we don't have in the router
	for id, source := range s.Sources {
		if !router.HasSource(id) {
			err := addSource(id, source)
			if err != nil {
				return err
			}
		}
	}

	// Remove any sources that are no longer in the state
	for _, source := range router.Sources {
		if !s.HasSource(source.SourceID) {
			err := removeSource(source.SourceID)
			if err != nil {
				return err
			}
		}
	}

	// Add any destinations we don't have in the router
	for id, dest := range s.Destinations {
		if !router.HasDestination(id) {
			err := addDestination(id, dest)
			if err != nil {
				return err
			}
		}
	}

	// Remove any destinations that are no longer in the state
	for _, dest := range router.Destinations {
		if !s.HasDestination(dest.DestinationID) {
			err := removeDestination(dest.DestinationID)
			if err != nil {
				return err
			}
		}
	}

	// Add any links we don't have in the router
	for _, route := range s.Routes {
		for destID := range route.Destinations {
			if !router.HasLink(route.SourceID, destID) {
				err := addLink(route.SourceID, destID)
				if err != nil {
					return err
				}
			}
		}

	}

	// Remove any links that are no longer in the state
	for _, link := range router.Links {
		if !s.HasRoute(link.SourceID) {
			err := removeLink(link.SourceID, link.DestinationID)
			if err != nil {
				return err
			}
		} else {
			r := s.GetRoute(link.SourceID)
			if _, ok := r.Destinations[link.DestinationID]; !ok {
				err := removeLink(link.SourceID, link.DestinationID)
				if err != nil {
					return err
				}
			}
		}
	}

	// Sync active state for all links
	for _, route := range s.Routes {
		for destID, dest := range route.Destinations {
			link := router.GetLink(route.SourceID, destID)
			link.Active = dest.Active
		}
	}

	return nil
}

func addSource(id string, source *state.Source) error {
	switch source.Type {
	case state.SourceTypeLRTMP:
		return addRTMPListener(id, source.Data)
	}

	return errors.New("Unrecognised source type")
}

func removeSource(id string) error {
	source := currentState.GetSource(id)
	if source != nil {
		switch source.Type {
		case state.SourceTypeLRTMP:
			return removeRTMPListener(id, source.Data)
		}

		return errors.New("Unrecognised source type")
	}
	return nil
}

func addRTMPListener(id string, data map[string]string) error {
	port, err := strconv.Atoi(data["port"])

	if err != nil {
		return err
	}

	var dip string
	var ok bool

	if dip, ok = data["ip"]; !ok || dip == "" {
		dip = "127.0.0.1"
	}

	ip := net.ParseIP(dip)

	if ip == nil {
		return fmt.Errorf("Failed to parse IP Address %v", dip)
	}

	ms, err := rtmp.AddListener(ip.String(), port, data["app_name"])

	if err != nil {
		return err
	}

	fmt.Printf("Added RTMP Listener with host %v port %v app %v\n", ip.String(), port, data["app_name"])

	router.AddSource(&router.Source{
		SourceID:    id,
		MediaStream: ms,
	})

	return nil
}

func removeRTMPListener(id string, data map[string]string) error {
	port, err := strconv.Atoi(data["port"])

	if err != nil {
		return err
	}

	var dip string
	var ok bool

	if dip, ok = data["ip"]; !ok || dip == "" {
		dip = "127.0.0.1"
	}

	ip := net.ParseIP(dip)

	if ip == nil {
		return fmt.Errorf("Failed to parse IP Address %v", dip)
	}

	err = rtmp.RemoveListener(ip.String(), port, data["app_name"])

	if err != nil {
		return err
	}

	err = router.RemoveSource(id)
	if err != nil {
		return err
	}

	return nil
}

func addDestination(id string, dest *state.Destination) error {
	switch dest.Type {
	case state.DestTypePRTMP:
		return addRTMPPusher(id, dest.Data)
	case state.DestTypeSRTMP:
		return addRTMPServer(id, dest.Data)
	case state.DestTypeFS:
		return addFileSink(id, dest.Data)
	}

	return errors.New("Unrecognised destination type")
}

func removeDestination(id string) error {
	dest := currentState.GetDestination(id)
	if dest != nil {
		switch dest.Type {
		case state.DestTypePRTMP:
			return removeRTMPPusher(id, dest.Data)
		case state.DestTypeSRTMP:
			return removeRTMPServer(id, dest.Data)
		case state.DestTypeFS:
			return removeFileSink(id, dest.Data)
		}

		return errors.New("Unrecognised destination type")
	}
	return nil
}

func addRTMPPusher(id string, data map[string]string) error {
	ms, err := rtmp.AddPusher(data["rtmp_url"])

	if err != nil {
		return err
	}

	fmt.Printf("Added RTMP Pusher with URL %v\n", data["rtmp_url"])

	router.AddDestination(&router.Destination{
		DestinationID: id,
		MediaStream:   ms,
	})

	return nil
}

func removeRTMPPusher(id string, data map[string]string) error {
	err := rtmp.RemovePusher(data["rtmp_url"])
	return err
}

func addRTMPServer(id string, data map[string]string) error {
	port, err := strconv.Atoi(data["port"])

	if err != nil {
		return err
	}

	var dip string
	var ok bool

	if dip, ok = data["ip"]; !ok || dip == "" {
		dip = "127.0.0.1"
	}

	ip := net.ParseIP(dip)

	if ip == nil {
		return fmt.Errorf("Failed to parse IP Address %v", dip)
	}

	ms, err := rtmp.AddListener(ip.String(), port, data["app_name"])

	if err != nil {
		return err
	}

	fmt.Printf("Added RTMP Listener with host %v port %v app %v\n", ip.String(), port, data["app_name"])

	router.AddSource(&router.Source{
		SourceID:    id,
		MediaStream: ms,
	})

	return nil
}

func removeRTMPServer(id string, data map[string]string) error {
	port, err := strconv.Atoi(data["port"])

	if err != nil {
		return err
	}

	var dip string
	var ok bool

	if dip, ok = data["ip"]; !ok || dip == "" {
		dip = "127.0.0.1"
	}

	ip := net.ParseIP(dip)

	if ip == nil {
		return fmt.Errorf("Failed to parse IP Address %v", dip)
	}

	err = rtmp.RemoveListener(ip.String(), port, data["app_name"])

	if err != nil {
		return err
	}

	err = router.RemoveSource(id)
	if err != nil {
		return err
	}

	return nil
}

func addFileSink(id string, data map[string]string) error {
	ms, err := rtmp.AddFileSink(data["file_path"])

	if err != nil {
		return err
	}

	fmt.Printf("Added File Sink with path %v\n", data["file_path"])

	router.AddDestination(&router.Destination{
		DestinationID: id,
		MediaStream:   ms,
	})

	return nil
}

func removeFileSink(id string, data map[string]string) error {
	err := rtmp.RemoveFileSink(data["file_path"])
	return err
}

func addLink(sourceID string, destinationID string) error {
	return router.AddLink(sourceID, destinationID)
}

func removeLink(sourceID string, destinationID string) error {
	for _, link := range router.Links {
		if link.SourceID == sourceID && link.DestinationID == destinationID {
			router.RemoveLink(sourceID, destinationID)
			break
		}
	}

	return nil
}
