package pusher

import (
	"bytes"
	"fmt"
	"log"
	"net"
	"net/url"
	"reflect"
	"strings"
	"sync"

	amf "code.justin.tv/video/goamf"
	"code.justin.tv/video/gortmp/pkg/rtmp"
	goctx "golang.org/x/net/context"
)

type TransactionHandler func(rtmp.Message) error
type TransactionMap struct {
	mu    sync.Mutex
	m     map[uint32]TransactionHandler
	txnid uint32
}

func NewTransactionMap() *TransactionMap {
	return &TransactionMap{
		m: make(map[uint32]TransactionHandler),
	}
}

func (tm *TransactionMap) New(h TransactionHandler) uint32 {
	tm.mu.Lock()
	defer tm.mu.Unlock()

	id := tm.txnid
	tm.txnid++
	tm.m[id] = h

	return id
}

func (tm *TransactionMap) Handle(id uint32, msg rtmp.Message) error {
	if h, ok := tm.m[id]; ok {
		delete(tm.m, id)
		return h(msg)
	}
	return nil
}

type RtmpPublisher struct {
	URL    *url.URL
	Conn   rtmp.BasicConn
	Stream rtmp.MediaStream
	tm     *TransactionMap
}

func (p *RtmpPublisher) ConnectResult(msg rtmp.Message) error {
	log.Printf("ConnectResult")

	cmd := rtmp.CreateStreamCommand{TransactionID: p.tm.New(p.CreateStreamResult)}
	return p.Send(cmd)
}

func (p *RtmpPublisher) CreateStreamResult(msg rtmp.Message) error {
	log.Printf("CreateStreamResult")

	result, ok := msg.(rtmp.ResultCommand)
	if !ok {
		return fmt.Errorf("Got invalid result command for CreateStream: %#v", msg)
	}

	log.Printf("CreateStreamResult: %#v", result)

	log.Printf("TypeOf(result.Info)=%s", reflect.TypeOf(result.Info).Name())

	streamID, ok := result.Info.(float64)
	if !ok {
		return fmt.Errorf("Invalid result commend for CreateStream: %#v", result)
	}

	path := strings.Split(p.URL.Path, "/")
	if len(path) == 0 {
		return fmt.Errorf("Unable to parse stream name: %s", p.URL.Path)
	}

	cmd := rtmp.PublishCommand{
		StreamID:      uint32(streamID),
		TransactionID: p.tm.New(p.PublishResult),
		Name:          path[len(path)-1],
	}

	return p.Send(cmd)
}

func (p *RtmpPublisher) PublishResult(msg rtmp.Message) error {
	log.Printf("PublishResult")

	result, ok := msg.(rtmp.OnStatusCommand)
	if !ok {
		return fmt.Errorf("Got invalid status command for Publish: %#v", msg)
	}

	go p.DoPublish(result.StreamID)

	return nil
}

func (p *RtmpPublisher) DoPublish(streamID uint32) {
	fmt.Println("DID PUBLISH")
	tags, err := p.Stream.Subscribe()

	if err != nil {
		log.Printf("Error subscribing to media stream %s", err)
		return
	}

	for {
		curTag, gotMsg := <-tags
		if !gotMsg {
			fmt.Println("stopped")
			break
		}
		raw := &rtmp.RawMessage{
			ChunkStreamID: 5,
			Timestamp:     curTag.Timestamp,
			Type:          curTag.Type,
			StreamID:      streamID,
			Data:          bytes.NewBuffer(curTag.Bytes),
		}

		if err := p.Conn.Write(raw); err != nil {
			log.Printf("Error during publish: %s", err)
			return
		}

		if err := p.Conn.Flush(); err != nil {
			log.Printf("Connection error during publish: %s", err)
			return
		}

	}
}

func (p *RtmpPublisher) Connect(RTMPurl string) {

	parsed, err := url.Parse(RTMPurl)
	if err != nil {
		log.Fatalf("Invalid rtmp url: %s", err)
	}

	conn, err := dial(parsed.Host)
	if err != nil {
		log.Fatal(err)
	}

	log.Printf("Connected to %s, performing handshake", "localhost")
	if _, err := rtmp.Handshake(goctx.Background(), conn); err != nil {
		log.Fatalf("Handshake failed: %s", err)
	}

	p.URL = parsed
	p.Conn = rtmp.NewBasicConn(conn)

}

func (p *RtmpPublisher) Send(msg rtmp.Message) error {
	raw, err := msg.RawMessage()
	if err != nil {
		return err
	}

	if err := p.Conn.Write(raw); err != nil {
		return err
	}

	return p.Conn.Flush()
}

func (p *RtmpPublisher) Push(ms rtmp.MediaStream) {

	fmt.Println("HERE IS THE DATA: %v", p.URL.String())
	p.Stream = ms
	p.tm = NewTransactionMap()

	tcURL := *p.URL
	tcURL.Path = "app"
	connectCmd := rtmp.ConnectCommand{
		TransactionID: p.tm.New(p.ConnectResult),
		Properties: amf.Object{
			"app":           "app",
			"flashVer":      "LNX 9,0,124,2",
			"tcUrl":         tcURL.String(),
			"fpad":          false,
			"capabilities":  15,
			"audioCodecs":   4071,
			"videoCodecs":   252,
			"videoFunction": 1,
		},
	}

	fmt.Println("Writing connect msg: %+v", connectCmd)
	if err := p.Send(connectCmd); err != nil {
		fmt.Println(err)
		return
	}

	fmt.Println("Writing chunk size")
	if err := p.Send(rtmp.SetChunkSizeMessage{Size: 4096}); err != nil {
		fmt.Println(err)
		return
	}

	for {
		raw, err := p.Conn.Read()
		if err != nil {
			fmt.Println(err)
			return
		}

		if err := p.HandleMessage(raw); err != nil {
			fmt.Println(err)
			return
		}
	}

}

func (p *RtmpPublisher) HandleMessage(raw *rtmp.RawMessage) error {
	msg, err := p.parseMessage(raw)
	if err != nil {
		return err
	}

	switch msg := msg.(type) {
	case rtmp.ResultCommand:
		return p.tm.Handle(msg.TransactionID, msg)
	case rtmp.ErrorCommand:
		return p.tm.Handle(msg.TransactionID, msg)
	case rtmp.OnStatusCommand:
		return p.tm.Handle(msg.TransactionID, msg)
	case *rtmp.RawMessage:
		raw := *msg
		raw.Data = nil
		log.Printf("Raw: %#v", raw)
	default:
		log.Printf("Received msg: %#v", msg)
	}
	return nil
}

func (p *RtmpPublisher) parseMessage(raw *rtmp.RawMessage) (rtmp.Message, error) {
	if raw.ChunkStreamID == rtmp.CS_ID_PROTOCOL_CONTROL {
		var msg rtmp.Message
		var err error

		if raw.Type == rtmp.USER_CONTROL_MESSAGE {
			msg, err = rtmp.ParseEvent(raw)
		} else {
			msg, err = rtmp.ParseControlMessage(raw)
		}

		return msg, err
	}

	switch raw.Type {
	case rtmp.COMMAND_AMF0:
		fallthrough
	case rtmp.COMMAND_AMF3:
		return rtmp.ParseCommand(raw)
	case rtmp.USER_CONTROL_MESSAGE:
		return rtmp.ParseEvent(raw)
	default:
		return raw, nil
	}
}

func dial(host string) (net.Conn, error) {
	if _, err := net.ResolveTCPAddr("tcp", host); err != nil {
		if _, ok := err.(net.InvalidAddrError); ok {
			host = net.JoinHostPort(host, "1935")
		}
	}

	return net.Dial("tcp", host)
}
