package router

import (
	"fmt"

	"code.justin.tv/video/gortmp/pkg/rtmp"
)

type Source struct {
	SourceID    string
	MediaStream rtmp.MediaStream
}

type Destination struct {
	DestinationID string
	MediaStream   rtmp.MediaStream
}

type Link struct {
	SourceID          string
	SourceMediaStream rtmp.MediaStream
	DestinationID     string
	DestMediaStream   rtmp.MediaStream
	Active            bool
	ch                *chan *rtmp.FlvTag
}

var Sources []*Source
var Destinations []*Destination
var Links []*Link

func AddSource(source *Source) {
	Sources = append(Sources, source)
}

func GetSource(id string) *Source {
	for _, source := range Sources {
		if source.SourceID == id {
			return source
		}
	}
	return nil
}

func RemoveSource(id string) error {
	for i, source := range Sources {
		if source.SourceID == id {
			Sources = append(Sources[:i], Sources[i+1:]...)
			break
		}
	}

	return nil
}

func HasSource(id string) bool {
	for _, source := range Sources {
		if source.SourceID == id {
			return true
		}
	}
	return false
}

func AddDestination(dest *Destination) {
	Destinations = append(Destinations, dest)
}

func GetDestination(id string) *Destination {
	for _, dest := range Destinations {
		if dest.DestinationID == id {
			return dest
		}
	}
	return nil
}

func RemoveDestination(id string) error {
	for i, dest := range Destinations {
		if dest.DestinationID == id {
			Destinations = append(Destinations[:i], Destinations[i+1:]...)
			break
		}
	}

	return nil
}

func HasDestination(id string) bool {
	for _, dest := range Destinations {
		if dest.DestinationID == id {
			return true
		}
	}
	return false
}

func AddLink(sourceID string, destinationID string) error {
	source := GetSource(sourceID)
	if source == nil {
		return fmt.Errorf("No source found for linking, source id: %v", sourceID)
	}

	dest := GetDestination(destinationID)
	if dest == nil {
		return fmt.Errorf("No destination found for linking, destination id: %v", destinationID)
	}

	link := &Link{
		SourceID:          sourceID,
		SourceMediaStream: source.MediaStream,
		DestinationID:     destinationID,
		DestMediaStream:   dest.MediaStream,
		Active:            false,
	}

	link.Setup()

	Links = append(Links, link)

	return nil
}

func GetLink(sourceID string, destID string) *Link {
	for _, link := range Links {
		if link.SourceID == sourceID && link.DestinationID == destID {
			return link
		}
	}
	return nil
}

func RemoveLink(sourceID string, destID string) error {
	for i, link := range Links {
		if link.SourceID == sourceID && link.DestinationID == destID {
			if link.Active {
				return fmt.Errorf("Cannot remove link as it is currently active sourceID: %v destinationID: %v", sourceID, destID)
			}

			link.Destroy()
			Links = append(Links[:i], Links[i+1:]...)
			break
		}
	}

	return nil
}

func HasLink(sourceID string, destID string) bool {
	for _, link := range Links {
		if link.SourceID == sourceID && link.DestinationID == destID {
			return true
		}
	}
	return false
}

func (ln *Link) Setup() {
	ch, err := ln.SourceMediaStream.Subscribe()
	if err != nil {
		fmt.Printf("Could not subscribe to mediastream with source ID: %v", ln.SourceID)
		return
	}

	ln.ch = &ch

	go func(l *Link) {
		for {
			tag, gotMsg := <-*l.ch
			if !gotMsg {
				fmt.Printf("Link closed: %v", l.SourceID)
				break
			}

			if l.Active {
				l.DestMediaStream.Publish(tag)
			}
		}
	}(ln)
}

func (ln *Link) Destroy() {
	if ln.ch == nil {
		return
	}

	ln.SourceMediaStream.Unsubscribe(*ln.ch)
}
