package rpc

import (
	"errors"
	"fmt"
	"log"
	"net"
	"net/http"
	"net/rpc"

	"code.justin.tv/event-engineering/covfefe/pkg/server/api"
)

// Covfefe RPC interface
type Covfefe struct {
	listener *api.Listener
}

// GenerateListener set up and return the RPC listener
func GenerateListener(port int32) *api.Listener {
	listener := new(api.Listener)
	covfefe := new(Covfefe)
	covfefe.listener = listener

	rpc.Register(covfefe)
	rpc.HandleHTTP()

	fmt.Printf("listening on port %v\n", port)

	l, e := net.Listen("tcp", fmt.Sprintf("localhost:%v", port))
	if e != nil {
		log.Fatal("listen error:", e)
	}
	go http.Serve(l, nil)

	return listener
}

/*=====================
		SOURCES
======================*/

// AddSourceRTMPL handler for AddSourceRTMPL RPC call
func (t *Covfefe) AddSourceRTMPL(args *api.AddSourceRTMPLArgs, reply *api.AddSourceRTMPLResponse) error {
	if t.listener.OnAddSourceRTMPL == nil {
		return errors.New("No callback for AddSourceRTMPL was found")
	}

	id, err := t.listener.OnAddSourceRTMPL(args.IP, args.Port, args.AppName)
	if err != nil {
		return err
	}

	*reply = api.AddSourceRTMPLResponse{
		ID:      id,
		IP:      args.IP,
		Port:    args.Port,
		AppName: args.AppName,
	}

	return nil
}

// RemoveSource handler for RemoveSource RPC call
func (t *Covfefe) RemoveSource(args *api.RemoveSourceArgs, reply *api.RemoveSourceResponse) error {
	if t.listener.OnRemoveSource == nil {
		return errors.New("No callback for RemoveSource was found")
	}

	id, err := t.listener.OnRemoveSource(args.ID)
	if err != nil {
		return err
	}

	*reply = api.RemoveSourceResponse{
		ID: id,
	}

	return nil
}

// ListSources handler for ListSources RPC call
func (t *Covfefe) ListSources(args *api.ListSourcesArgs, reply *api.ListSourcesResponse) error {
	if t.listener.OnListSources == nil {
		return errors.New("No callback for ListSources was found")
	}

	sources, err := t.listener.OnListSources()
	if err != nil {
		return err
	}

	*reply = api.ListSourcesResponse{
		Sources: sources,
	}

	return nil
}

/*=====================
	  DESTINATIONS
======================*/

// AddDestSRTMP handler for AddDestSRTMP RPC call
func (t *Covfefe) AddDestSRTMP(args *api.AddDestSRTMPArgs, reply *api.AddDestSRTMPResponse) error {
	if t.listener.OnAddDestSRTMP == nil {
		return errors.New("No callback for AddDestSRTMP was found")
	}

	id, err := t.listener.OnAddDestSRTMP(args.IP, args.Port, args.AppName)
	if err != nil {
		return err
	}

	*reply = api.AddDestSRTMPResponse{
		ID:      id,
		IP:      args.IP,
		Port:    args.Port,
		AppName: args.AppName,
	}

	return nil
}

// AddDestPRTMP handler for AddDestPRTMP RPC call
func (t *Covfefe) AddDestPRTMP(args *api.AddDestPRTMPArgs, reply *api.AddDestPRTMPResponse) error {
	if t.listener.OnAddDestPRTMP == nil {
		return errors.New("No callback for AddDestPRTMP was found")
	}

	id, err := t.listener.OnAddDestPRTMP(args.URL)
	if err != nil {
		return err
	}

	*reply = api.AddDestPRTMPResponse{
		ID:  id,
		URL: args.URL,
	}

	return nil
}

// AddDestFS handler for AddDestFS RPC call
func (t *Covfefe) AddDestFS(args *api.AddDestFSArgs, reply *api.AddDestFSResponse) error {
	if t.listener.OnAddDestFS == nil {
		return errors.New("No callback for OnAddDestFS was found")
	}

	id, err := t.listener.OnAddDestFS(args.FilePath)
	if err != nil {
		return err
	}

	*reply = api.AddDestFSResponse{
		ID:       id,
		FilePath: args.FilePath,
	}

	return nil
}

// RemoveDestination handler for RemoveDestination RPC call
func (t *Covfefe) RemoveDestination(args *api.RemoveDestinationArgs, reply *api.RemoveDestinationResponse) error {
	if t.listener.OnRemoveDestination == nil {
		return errors.New("No callback for RemoveDestination was found")
	}

	id, err := t.listener.OnRemoveDestination(args.ID)
	if err != nil {
		return err
	}

	*reply = api.RemoveDestinationResponse{
		ID: id,
	}

	return nil
}

// ListDestinations handler for ListDestinations RPC call
func (t *Covfefe) ListDestinations(args *api.ListDestinationsArgs, reply *api.ListDestinationsResponse) error {
	if t.listener.OnListDestinations == nil {
		return errors.New("No callback for ListDestinations was found")
	}

	destinations, err := t.listener.OnListDestinations()
	if err != nil {
		return err
	}

	*reply = api.ListDestinationsResponse{
		Destinations: destinations,
	}

	return nil
}

/*=====================
		 ROUTES
======================*/

// AddRoute handler for AddRoute RPC call
func (t *Covfefe) AddRoute(args *api.AddRouteArgs, reply *api.AddRouteResponse) error {
	if t.listener.OnAddRoute == nil {
		return errors.New("No callback for AddRoute was found")
	}

	err := t.listener.OnAddRoute(args.SourceID)
	if err != nil {
		return err
	}

	*reply = api.AddRouteResponse{
		SourceID: args.SourceID,
	}

	return nil
}

// LinkRoute handler for LinkRoute RPC call
func (t *Covfefe) LinkRoute(args *api.LinkRouteArgs, reply *api.LinkRouteResponse) error {
	if t.listener.OnLinkRoute == nil {
		return errors.New("No callback for LinkRoute was found")
	}

	err := t.listener.OnLinkRoute(args.SourceID, args.DestinationID)
	if err != nil {
		return err
	}

	*reply = api.LinkRouteResponse{
		SourceID:      args.SourceID,
		DestinationID: args.DestinationID,
	}

	return nil
}

// UnlinkRoute handler for UnlinkRoute RPC call
func (t *Covfefe) UnlinkRoute(args *api.UnlinkRouteArgs, reply *api.UnlinkRouteResponse) error {
	if t.listener.OnUnlinkRoute == nil {
		return errors.New("No callback for UnlinkRoute was found")
	}

	err := t.listener.OnUnlinkRoute(args.SourceID, args.DestinationID)
	if err != nil {
		return err
	}

	*reply = api.UnlinkRouteResponse{
		SourceID:      args.SourceID,
		DestinationID: args.DestinationID,
	}

	return nil
}

// ActivateRoute handler for ActivateRoute RPC call
func (t *Covfefe) ActivateRoute(args *api.ActivateRouteArgs, reply *api.ActivateRouteResponse) error {
	if t.listener.OnActivateRoute == nil {
		return errors.New("No callback for ActivateRoute was found")
	}

	err := t.listener.OnActivateRoute(args.SourceID, args.DestinationID)
	if err != nil {
		return err
	}

	*reply = api.ActivateRouteResponse{
		SourceID:      args.SourceID,
		DestinationID: args.DestinationID,
	}

	return nil
}

// DeactivateRoute handler for DeactivateRoute RPC call
func (t *Covfefe) DeactivateRoute(args *api.DeactivateRouteArgs, reply *api.DeactivateRouteResponse) error {
	if t.listener.OnDeactivateRoute == nil {
		return errors.New("No callback for DeactivateRoute was found")
	}

	err := t.listener.OnDeactivateRoute(args.SourceID, args.DestinationID)
	if err != nil {
		return err
	}

	*reply = api.DeactivateRouteResponse{
		SourceID:      args.SourceID,
		DestinationID: args.DestinationID,
	}

	return nil
}

// ListRoutes handler for ListRoutes RPC call
func (t *Covfefe) ListRoutes(args *api.ListRoutesArgs, reply *api.ListRoutesResponse) error {
	if t.listener.OnListRoutes == nil {
		return errors.New("No callback for ListRoutes was found")
	}

	routes, err := t.listener.OnListRoutes()
	if err != nil {
		return err
	}

	*reply = api.ListRoutesResponse{
		Routes: routes,
	}

	return nil
}

// RemoveRoute handler for RemopveRoute RPC call
func (t *Covfefe) RemoveRoute(args *api.RemoveRouteArgs, reply *api.RemoveRouteResponse) error {
	if t.listener.OnRemoveRoute == nil {
		return errors.New("No callback for RemoveRoute was found")
	}

	err := t.listener.OnRemoveRoute(args.SourceID)
	if err != nil {
		return err
	}

	*reply = api.RemoveRouteResponse{
		SourceID: args.SourceID,
	}

	return nil
}
