package rtmp

import (
	"context"
	"errors"
	"fmt"
	"log"

	flv "code.justin.tv/video/goflv"
	gortmp "code.justin.tv/video/gortmp/pkg/rtmp"
)

const (
	fsRoot string = "/opt/something"
)

var fileSinks []*RTMPFileSink

type RTMPFileSink struct {
	filePath string
	ms       gortmp.MediaStream
	ch       chan *gortmp.FlvTag
}

func AddFileSink(filePath string) (gortmp.MediaStream, error) {
	// Check if there is already a sink for this filePath, otherwise create it
	for _, p := range fileSinks {
		if p.filePath == filePath {
			return nil, errors.New("There is already an RTMP file sink for this path")
		}
	}

	fs, err := createFileSink(filePath)

	if err != nil {
		log.Printf("Error creating RTMP file sink %v", err)
		return nil, err
	}

	if len(fileSinks) == 0 {
		fileSinks = make([]*RTMPFileSink, 0)
	}
	fileSinks = append(fileSinks, fs)

	return fs.ms, nil
}

func RemoveFileSink(filePath string) error {
	var idx int
	var fs *RTMPFileSink

	for i, s := range fileSinks {
		if s.filePath == filePath {
			idx = i
			fs = s
			break
		}
	}

	if fs == nil {
		return fmt.Errorf("No RTMP file sink found with path %v", filePath)
	}

	// Not sure what else we need to do here
	close(fs.ch)

	fileSinks = append(fileSinks[:idx], fileSinks[idx+1:]...)

	return nil
}

func createFileSink(filePath string) (*RTMPFileSink, error) {
	fs := &RTMPFileSink{
		filePath: filePath,
	}

	ms := gortmp.NewMediaStream(context.TODO(), filePath)
	fs.ms = ms

	filename := /*fsRoot +*/ filePath
	flvFile, err := flv.CreateFile(filename)

	if err != nil {
		return nil, fmt.Errorf("Could not create file %v", err)
	}

	tagCh, err := fs.ms.Subscribe()
	fs.ch = tagCh

	if err != nil {
		return nil, fmt.Errorf("Could not subscribe to media stream %v", err)
	}

	go func(tagCh chan *gortmp.FlvTag, sink *RTMPFileSink) {
		for {
			tag, ok := <-tagCh
			if !ok {
				flvFile.Close()
				break
			}
			flvFile.WriteTag(tag.Bytes, tag.Type, tag.Timestamp)
		}
	}(tagCh, fs)

	return fs, nil
}
