package rtmp

import (
	"context"
	"errors"
	"fmt"
	"log"
	"net"
	"strconv"

	rtmpctx "code.justin.tv/video/gortmp/pkg/context"
	gortmp "code.justin.tv/video/gortmp/pkg/rtmp"
	goctx "golang.org/x/net/context"
)

func AddListener(host string, port int, appName string) (gortmp.MediaStream, error) {
	// Check if there is already a listener for this host/port, otherwise create it
	var ln *RTMPListener
	for _, l := range listeners {
		if l.host == host && l.port == port {
			ln = l
			break
		}
	}
	var err error

	if ln == nil {
		ln, err = createListener(host, port)

		if err != nil {
			log.Printf("Error creating RTMP listener %v", err)
			return nil, err
		}

		if len(listeners) == 0 {
			listeners = make([]*RTMPListener, 0)
		}
		listeners = append(listeners, ln)
	}

	if _, ok := ln.rtmpData[appName]; ok {
		return nil, errors.New("There is already an RTMP listener for this host/port/app combo")
	}

	ms := gortmp.NewMediaStream(context.TODO(), appName)

	ln.rtmpData[appName] = &RTMPStream{
		destMediaStream: ms,
	}

	return ms, nil
}

func RemoveListener(host string, port int, appName string) error {
	var ln *RTMPListener
	var lnIdx int
	for i, l := range listeners {
		if l.host == host && l.port == port {
			ln = l
			lnIdx = i
			break
		}
	}

	if ln == nil {
		return fmt.Errorf("Cannot remove listener, no listener found with host %v and port %v", host, port)
	}

	rtmpStream, ok := ln.rtmpData[appName]

	if !ok {
		return fmt.Errorf("Cannot remove listener, no app name %v found on listener", appName)
	}

	rtmpStream.UnlinkMediaStreams()
	delete(ln.rtmpData, appName)

	if len(ln.rtmpData) == 0 {
		ln.listener.Close()
		listeners = append(listeners[:lnIdx], listeners[lnIdx+1:]...)
	}

	return nil
}

type rtmpListenerEventHandler struct {
	mediaServer *gortmp.MediaServer
	c           goctx.Context
}

func getListener(host string, port int) *RTMPListener {
	for _, l := range listeners {
		if l.host == host && l.port == port {
			return l
		}
	}

	return nil
}

func (h *rtmpListenerEventHandler) OnMediaStreamCreated(c goctx.Context, ms gortmp.MediaStream) {
	host, port, appName := h.getRtmpInfo(c)
	ln := getListener(host, port)

	if ln == nil {
		// This should be impossible
		log.Fatalf("No Listener found for mediastream %v:%v\n", host, port)
	}

	rtmpStream, ok := ln.rtmpData[appName]

	if !ok {
		log.Printf("No Listener found for mediastream %v:%v\n", host, port)
		return
	}

	rtmpStream.sourceMediaStream = ms
	rtmpStream.LinkMediaStreams()
}

func (h *rtmpListenerEventHandler) getRtmpInfo(ctx goctx.Context) (host string, port int, app string) {
	addr, _ := rtmpctx.GetListenAddr(ctx)
	host, sport, _ := net.SplitHostPort(addr.String())
	port, _ = strconv.Atoi(sport)
	app = h.c.Value(appNameKey).(string)

	return
}

func (h *rtmpListenerEventHandler) OnMediaStreamDestroyed(c goctx.Context, ms gortmp.MediaStream) {
	host, port, appName := h.getRtmpInfo(c)
	ln := getListener(host, port)

	if ln == nil {
		// This should be impossible
		log.Fatalf("No Listener found for mediastream %v:%v\n", host, port)
	}

	rtmpStream, ok := ln.rtmpData[appName]

	if !ok {
		log.Printf("No Listener found for mediastream %v:%v\n", host, port)
		return
	}

	rtmpStream.UnlinkMediaStreams()
	ms.Close()
}

func (h *rtmpListenerEventHandler) Handle(ctx goctx.Context, r gortmp.Receiver, msg gortmp.Message) error {
	switch msg := msg.(type) {
	case gortmp.ConnectCommand:
		err := h.onConnect(ctx, msg)
		if err != nil {
			return err
		}
	}

	return r.Handle(msg)
}

func (h *rtmpListenerEventHandler) onConnect(ctx goctx.Context, cmd gortmp.ConnectCommand) error {
	// Add the app name to the context so we have it later
	if app, ok := cmd.Properties["app"].(string); ok {
		h.c = goctx.WithValue(h.c, appNameKey, app)
	}

	// Check to see if we're expecting anything to this app name
	host, port, appName := h.getRtmpInfo(ctx)
	ln := getListener(host, port)

	if ln == nil {
		// This should be impossible
		log.Fatalf("No Listener found %v:%v\n", host, port)
	}

	if _, ok := ln.rtmpData[appName]; !ok {
		return gortmp.ErrConnectRejected(fmt.Errorf("invalid app: %v", appName))
	}

	return nil
}
