package rtmp

import (
	"context"
	"errors"
	"fmt"
	"log"

	"code.justin.tv/event-engineering/covfefe/pkg/server/pusher"
	gortmp "code.justin.tv/video/gortmp/pkg/rtmp"
)

var pushers []*RTMPPusher

type RTMPPusher struct {
	url string
	ms  gortmp.MediaStream
	pub *pusher.RtmpPublisher
}

func AddPusher(rtmpUrl string) (gortmp.MediaStream, error) {
	// Check if there is already a pusher for this URL, otherwise create it
	for _, p := range pushers {
		if p.url == rtmpUrl {
			return nil, errors.New("There is already an RTMP pusher for this URL")
		}
	}

	ps, err := createPusher(rtmpUrl)

	if err != nil {
		log.Printf("Error creating RTMP pusher %v", err)
		return nil, err
	}

	if len(pushers) == 0 {
		pushers = make([]*RTMPPusher, 0)
	}
	pushers = append(pushers, ps)

	return ps.ms, nil
}

func RemovePusher(rtmpUrl string) error {
	var idx int
	var ps *RTMPPusher

	for i, p := range pushers {
		if p.url == rtmpUrl {
			idx = i
			ps = p
			break
		}
	}

	if ps == nil {
		return fmt.Errorf("No RTMP pusher found with URL %v", rtmpUrl)
	}

	// Not sure what else we need to do here
	ps.pub.Conn.Flush()
	ps.pub.Conn.Close()

	pushers = append(pushers[:idx], pushers[idx+1:]...)

	return nil
}

func createPusher(rtmpUrl string) (*RTMPPusher, error) {
	ps := &RTMPPusher{
		url: rtmpUrl,
	}

	ms := gortmp.NewMediaStream(context.TODO(), rtmpUrl)
	ps.ms = ms

	p := pusher.RtmpPublisher{}
	ps.pub = &p

	// Intercept the first tag and call Push on the pusher, then resend the tag to the mediastream so it gets picked up by the pusher
	go func(rtmpPusher *RTMPPusher) {
		ch, err := rtmpPusher.ms.Subscribe()
		if err != nil {
			log.Printf("Error subscribing to mediastream %v", err)
			return
		}

		tag := <-ch
		rtmpPusher.ms.Unsubscribe(ch)

		p.Connect(rtmpUrl)
		p.Push(rtmpPusher.ms)

		go func(p *RTMPPusher, t *gortmp.FlvTag) {
			p.ms.Publish(t)
		}(rtmpPusher, tag)
	}(ps)

	return ps, nil
}
