package rtmp

import (
	"fmt"
	"log"
	"net"
	"strconv"
	"time"

	"code.justin.tv/video/goingest/pkg/util"
	gortmp "code.justin.tv/video/gortmp/pkg/rtmp"
	goctx "golang.org/x/net/context"
)

type key int

const (
	appNameKey key = iota
)

type RTMPStream struct {
	destMediaStream   gortmp.MediaStream
	sourceMediaStream gortmp.MediaStream
	isLinked          bool
	ch                *chan *gortmp.FlvTag
}

var listeners []*RTMPListener

type RTMPListener struct {
	host     string
	port     int
	listener net.Listener
	rtmpData map[string]*RTMPStream
}

type rtmpListenerHandler struct{}

func (ps *rtmpListenerHandler) ServeRTMP(ctx goctx.Context, conn net.Conn) error {
	rtmpListenerEventHandler := &rtmpListenerEventHandler{
		c: ctx,
	}
	ms := gortmp.NewMediaServer(rtmpListenerEventHandler)
	rtmpListenerEventHandler.mediaServer = ms

	go ms.ServeRTMP(ctx, conn)

	return nil
}

func createListener(host string, port int) (*RTMPListener, error) {
	rtmpserver := gortmp.NewServer(gortmp.ServerConfig{
		Handler: &rtmpListenerHandler{},
	})
	log.Println(net.JoinHostPort(host, strconv.Itoa(port)))
	ta, err := net.ResolveTCPAddr("tcp", net.JoinHostPort(host, strconv.Itoa(port)))

	if err != nil {
		return &RTMPListener{}, err
	}

	ln, err := net.ListenTCP("tcp", ta)
	if err != nil {
		return &RTMPListener{}, err
	}

	mLn := util.NewMultiListener(ln, 10*time.Second)
	d := mLn.RtmpListener()

	go func(l net.Listener) {
		err := rtmpserver.Serve(l)
		fmt.Printf("Done listening on %s: %s\n", mLn.RtmpListener().Addr(), err)
	}(d)

	return &RTMPListener{
		host:     host,
		port:     port,
		rtmpData: make(map[string]*RTMPStream),
		listener: d,
	}, nil
}

func (s *RTMPStream) LinkMediaStreams() {
	go func(rs *RTMPStream) {
		sourceChannel, err := rs.sourceMediaStream.Subscribe()
		if err != nil {
			log.Printf("Error subscribing to MediaStream %v\n", err)
			return
		}

		rs.ch = &sourceChannel
		rs.isLinked = true

		for {
			if !rs.isLinked {
				break
			}
			tag := <-*rs.ch
			if tag != nil {
				rs.destMediaStream.Publish(tag)
			}
		}
	}(s)
}

func (s *RTMPStream) UnlinkMediaStreams() {
	s.isLinked = false
	s.sourceMediaStream.Unsubscribe(*s.ch)
	s.ch = nil
}
