package rtmp

import (
	"context"
	"errors"
	"fmt"
	"log"
	"net"
	"strconv"
	"time"

	"code.justin.tv/video/goingest/pkg/util"
	rtmpctx "code.justin.tv/video/gortmp/pkg/context"
	gortmp "code.justin.tv/video/gortmp/pkg/rtmp"
	goctx "golang.org/x/net/context"
)

var servers []*RTMPServer

type RTMPServer struct {
	host        string
	port        int
	listener    net.Listener
	mediaServer *gortmp.MediaServer
	ctx         goctx.Context
	rtmpData    map[string]*RTMPStream
}

func AddServer(host string, port int, appName string) (gortmp.MediaStream, error) {
	return nil, errors.New("Unsupported")

	// Check if there is already a listener for this host/port, otherwise create it
	var sv *RTMPServer
	for _, s := range servers {
		if s.host == host && s.port == port {
			sv = s
			break
		}
	}
	var err error

	if sv == nil {
		sv, err = createServer(host, port)

		if err != nil {
			log.Printf("Error creating RTMP server %v", err)
			return nil, err
		}

		if len(servers) == 0 {
			servers = make([]*RTMPServer, 0)
		}
		servers = append(servers, sv)
	}

	if _, ok := sv.rtmpData[appName]; ok {
		return nil, errors.New("There is already an RTMP listener for this host/port/app combo")
	}

	ctx := context.TODO()
	ms := gortmp.NewMediaStream(ctx, appName)

	sv.mediaServer.AddStream(ctx, ms)

	sv.rtmpData[appName] = &RTMPStream{
		destMediaStream: ms,
	}

	sv.rtmpData[appName].LinkMediaStreams()

	return ms, nil
}

func RemoveServer(host string, port int, appName string) error {
	return errors.New("Unsupported")

	var sv *RTMPServer
	var svIdx int
	for i, s := range servers {
		if s.host == host && s.port == port {
			sv = s
			svIdx = i
			break
		}
	}

	if sv == nil {
		return fmt.Errorf("Cannot remove server, no server found with host %v and port %v", host, port)
	}

	rtmpStream, ok := sv.rtmpData[appName]

	if !ok {
		return fmt.Errorf("Cannot remove server, no app name %v found on server", appName)
	}

	rtmpStream.UnlinkMediaStreams()
	sv.mediaServer.DeleteStream(sv.ctx, appName)
	delete(sv.rtmpData, appName)

	if len(sv.rtmpData) == 0 {
		sv.listener.Close()
		servers = append(servers[:svIdx], servers[svIdx+1:]...)
	}

	return nil
}

type rtmpServerHandler struct{}

func createServer(host string, port int) (*RTMPServer, error) {
	ctx := goctx.TODO()

	rtmpServerEventHandler := &rtmpServerEventHandler{
		c: ctx,
	}
	ms := gortmp.NewMediaServer(rtmpServerEventHandler)
	rtmpServerEventHandler.mediaServer = ms

	rtmpserver := gortmp.NewServer(gortmp.ServerConfig{
		Handler: ms,
		Context: ctx,
	})

	log.Println(net.JoinHostPort(host, strconv.Itoa(port)))
	ta, err := net.ResolveTCPAddr("tcp", net.JoinHostPort(host, strconv.Itoa(port)))

	if err != nil {
		return &RTMPServer{}, err
	}

	ln, err := net.ListenTCP("tcp", ta)
	if err != nil {
		return &RTMPServer{}, err
	}

	mLn := util.NewMultiListener(ln, 10*time.Second)
	d := mLn.RtmpListener()

	go func(l net.Listener) {
		err := rtmpserver.Serve(l)
		fmt.Printf("Done listening on %s: %s\n", mLn.RtmpListener().Addr(), err)
	}(d)

	return &RTMPServer{
		host:     host,
		port:     port,
		rtmpData: make(map[string]*RTMPStream),
		listener: d,
	}, nil
}

type rtmpServerEventHandler struct {
	mediaServer *gortmp.MediaServer
	c           goctx.Context
}

func getServer(host string, port int) *RTMPServer {
	for _, l := range servers {
		if l.host == host && l.port == port {
			return l
		}
	}

	return nil
}

func (h *rtmpServerEventHandler) OnMediaStreamCreated(c goctx.Context, ms gortmp.MediaStream) {

	host, port, appName := h.getRtmpInfo(c)
	ln := getListener(host, port)

	if ln == nil {
		// This should be impossible
		log.Fatalf("No Listener found for mediastream %v:%v\n", host, port)
	}

	rtmpStream, ok := ln.rtmpData[appName]

	if !ok {
		log.Printf("No Listener found for mediastream %v:%v\n", host, port)
		return
	}

	rtmpStream.sourceMediaStream = ms

}

func (h *rtmpServerEventHandler) getRtmpInfo(ctx goctx.Context) (host string, port int, app string) {
	addr, _ := rtmpctx.GetListenAddr(ctx)
	host, sport, _ := net.SplitHostPort(addr.String())
	port, _ = strconv.Atoi(sport)
	app = h.c.Value(appNameKey).(string)

	return
}

func (h *rtmpServerEventHandler) OnMediaStreamDestroyed(c goctx.Context, ms gortmp.MediaStream) {
	host, port, appName := h.getRtmpInfo(c)
	ln := getListener(host, port)

	if ln == nil {
		// This should be impossible
		log.Fatalf("No Listener found for mediastream %v:%v\n", host, port)
	}

	rtmpStream, ok := ln.rtmpData[appName]

	if !ok {
		log.Printf("No Listener found for mediastream %v:%v\n", host, port)
		return
	}

	rtmpStream.UnlinkMediaStreams()
	ms.Close()
}

func (h *rtmpServerEventHandler) Handle(ctx goctx.Context, r gortmp.Receiver, msg gortmp.Message) error {
	switch msg := msg.(type) {
	case gortmp.ConnectCommand:
		err := h.onConnect(ctx, msg)
		if err != nil {
			return err
		}
	}

	return r.Handle(msg)
}

func (h *rtmpServerEventHandler) onConnect(ctx goctx.Context, cmd gortmp.ConnectCommand) error {
	// Add the app name to the context so we have it later
	if app, ok := cmd.Properties["app"].(string); ok {
		h.c = goctx.WithValue(h.c, appNameKey, app)
	}

	// Check to see if we're expecting anything to this app name
	host, port, appName := h.getRtmpInfo(ctx)
	ln := getServer(host, port)

	if ln == nil {
		// This should be impossible
		log.Fatalf("No Server found %v:%v\n", host, port)
	}

	if _, ok := ln.rtmpData[appName]; !ok {
		return gortmp.ErrConnectRejected(fmt.Errorf("invalid app: %v", appName))
	}

	return nil
}
