package state

import (
	"fmt"
	"strconv"

	"code.justin.tv/event-engineering/covfefe/pkg/server/uid"
)

// Source Types
const SourceTypeLRTMP = 1

// Destination Types
const DestTypeSRTMP = 1
const DestTypePRTMP = 2
const DestTypeFS = 3

// State state object for the server
type State struct {
	Sources      map[string]*Source      `json:"sources"`
	Destinations map[string]*Destination `json:"destinations"`
	Routes       []Route                 `json:"routes"`
}

// Source database source
type Source struct {
	Type int               `json:"type"`
	Data map[string]string `json:"data"`
}

// Destination database source
type Destination struct {
	Type int               `json:"type"`
	Data map[string]string `json:"data"`
}

// Route database source
type Route struct {
	SourceID     string                      `json:"source"`
	Destinations map[string]*RouteDestParams `json:"destinations"`
}

// RouteDestParams status of route destination
type RouteDestParams struct {
	Active bool `json:"active"`
}

/*=====================
		SOURCES
======================*/

// AddSourceRTMPL validates and adds an RTMP listener source to the state object
func (state *State) AddSourceRTMPL(ip string, port int, appName string) (id string, e error) {
	for id, source := range state.Sources {
		if source.Type == SourceTypeLRTMP {
			parsedPort, _ := strconv.Atoi(source.Data["port"])
			if parsedPort == port && source.Data["ip"] == ip && source.Data["app_name"] == appName {
				return "", fmt.Errorf("Source [%v] already exists with the specified ip, port and app name", id)
			}
		}
	}

	if state.Sources == nil {
		state.Sources = make(map[string]*Source)
	}

	for true {
		id = uid.Generate()
		if _, ok := state.Sources[id]; !ok {
			state.Sources[id] = &Source{
				Type: SourceTypeLRTMP,
				Data: map[string]string{
					"ip":       ip,
					"port":     strconv.Itoa(port),
					"app_name": appName,
				},
			}
			break
		}
	}

	return id, nil
}

// RemoveSource validate and remove source from the state
func (state *State) RemoveSource(ID string) error {
	if _, ok := state.Sources[ID]; !ok {
		return fmt.Errorf("No source exists with id %v", ID)
	}

	for _, route := range state.Routes {
		if route.SourceID == ID {
			return fmt.Errorf("Cannot remove source as it exists in route %v", ID)
		}
	}

	delete(state.Sources, ID)

	return nil
}

// HasSource checks to see if a given source ID is present in the state
func (state *State) HasSource(id string) bool {
	_, ok := state.Sources[id]
	return ok
}

// GetSource return source if present in state
func (state *State) GetSource(id string) *Source {
	if src, ok := state.Sources[id]; ok {
		return src
	}
	return nil
}

// HasDestination checks to see if a given destination ID is present in the state
func (state *State) HasDestination(id string) bool {
	_, ok := state.Destinations[id]
	return ok
}

// GetDestination return destination if present in state
func (state *State) GetDestination(id string) *Destination {
	if dest, ok := state.Destinations[id]; ok {
		return dest
	}
	return nil
}

// HasRoute checks to see if a given route ID is present in the state
func (state *State) HasRoute(id string) bool {
	for _, route := range state.Routes {
		if route.SourceID == id {
			return true
		}
	}
	return false
}

// GetRoute return route if present in state
func (state *State) GetRoute(id string) *Route {
	for _, route := range state.Routes {
		if route.SourceID == id {
			return &route
		}
	}
	return nil
}

/*=====================
	  DESTINATIONS
======================*/

// AddDestSRTMP validates and adds a Serve RTMP destination to the state object
func (state *State) AddDestSRTMP(ip string, port int, appName string) (id string, e error) {
	for id, dest := range state.Destinations {
		if dest.Type == DestTypeSRTMP {
			parsedPort, _ := strconv.Atoi(dest.Data["port"])
			if parsedPort == port && dest.Data["ip"] == ip && dest.Data["app_name"] == appName {
				return "", fmt.Errorf("Destination [%v] already exists with the specified ip, port and app name", id)
			}
		}
	}

	if state.Destinations == nil {
		state.Destinations = make(map[string]*Destination)
	}

	for true {
		id = uid.Generate()
		if _, ok := state.Destinations[id]; !ok {
			state.Destinations[id] = &Destination{
				Type: DestTypeSRTMP,
				Data: map[string]string{
					"ip":       ip,
					"port":     strconv.Itoa(port),
					"app_name": appName,
				},
			}
			break
		}
	}

	return id, nil
}

// AddDestPRTMP validates and adds a Push RTMP destination to the state object
func (state *State) AddDestPRTMP(URL string) (id string, e error) {
	for id, dest := range state.Destinations {
		if dest.Type == DestTypePRTMP && dest.Data["rtmp_url"] == URL {
			return "", fmt.Errorf("Destination [%v] already exists with the specified URL", id)
		}
	}

	if state.Destinations == nil {
		state.Destinations = make(map[string]*Destination)
	}

	for true {
		id = uid.Generate()
		if _, ok := state.Destinations[id]; !ok {
			state.Destinations[id] = &Destination{
				Type: DestTypePRTMP,
				Data: map[string]string{
					"rtmp_url": URL,
				},
			}
			break
		}
	}

	return id, nil
}

// AddDestFS validates and adds a File Sink RTMP destination to the state object
func (state *State) AddDestFS(filePath string) (id string, e error) {
	for id, dest := range state.Destinations {
		if dest.Type == DestTypeFS && dest.Data["file_path"] == filePath {
			return "", fmt.Errorf("Destination [%v] already exists with the specified file path", id)
		}
	}

	if state.Destinations == nil {
		state.Destinations = make(map[string]*Destination)
	}

	for true {
		id = uid.Generate()
		if _, ok := state.Destinations[id]; !ok {
			state.Destinations[id] = &Destination{
				Type: DestTypeFS,
				Data: map[string]string{
					"file_path": filePath,
				},
			}
			break
		}
	}

	return id, nil
}

// RemoveDestination validate and remove destination from the state
func (state *State) RemoveDestination(ID string) error {
	if _, ok := state.Destinations[ID]; !ok {
		return fmt.Errorf("No destination exists with id %v", ID)
	}

	for _, route := range state.Routes {
		for destID := range route.Destinations {
			if destID == ID {
				return fmt.Errorf("Cannot remove destination as it exists in route %v", route.SourceID)
			}
		}
	}

	delete(state.Destinations, ID)

	return nil
}

/*=====================
		 ROUTES
======================*/

// AddRoute validates and adds a Route for the given source to the state object
func (state *State) AddRoute(sourceID string) (e error) {
	if _, ok := state.Sources[sourceID]; !ok {
		return fmt.Errorf("Could not find source with ID %v", sourceID)
	}

	for _, route := range state.Routes {
		if route.SourceID == sourceID {
			return fmt.Errorf("Source %v is lready associated to a route", sourceID)
		}
	}

	if state.Routes == nil {
		state.Routes = make([]Route, 0)
	}

	route := Route{
		SourceID: sourceID,
	}

	state.Routes = append(state.Routes, route)

	return nil
}

// LinkRoute validates and links a Destination to the given source in the state object
func (state *State) LinkRoute(sourceID string, destinationID string) (e error) {
	var route *Route
	var destLinkedSource string

	for i, r := range state.Routes {
		if r.SourceID == sourceID {
			route = &state.Routes[i]
		}

		for id := range r.Destinations {
			if id == destinationID {
				destLinkedSource = r.SourceID
				break
			}
		}
	}

	if route.SourceID == "" {
		return fmt.Errorf("Could not find Route with Source ID %v", sourceID)
	}

	if _, ok := state.Destinations[destinationID]; !ok {
		return fmt.Errorf("Could not find destination with ID %v", destinationID)
	}

	if destLinkedSource != "" {
		return fmt.Errorf("Destination is already linked to source with ID %v", destLinkedSource)
	}

	if route.Destinations == nil {
		route.Destinations = make(map[string]*RouteDestParams)
	}

	route.Destinations[destinationID] = &RouteDestParams{}

	return nil
}

// UnlinkRoute validates and unlinks a Destination from the given source in the state object
func (state *State) UnlinkRoute(sourceID string, destinationID string) (e error) {
	var route *Route

	for i, r := range state.Routes {
		if r.SourceID == sourceID {
			route = &state.Routes[i]
			break
		}
	}

	if route.SourceID == "" {
		return fmt.Errorf("Could not find Route with Source ID %v", sourceID)
	}

	if _, ok := state.Destinations[destinationID]; !ok {
		return fmt.Errorf("Could not find destination with ID %v", destinationID)
	}

	if _, ok := route.Destinations[destinationID]; !ok {
		return fmt.Errorf("Destination %v is not associated to route with source ID %v", destinationID, sourceID)
	}

	if route.Destinations[destinationID].Active {
		return fmt.Errorf("Cannot remove Destination %v as it is currently active", destinationID)
	}

	delete(route.Destinations, destinationID)

	return nil
}

// ActivateRoute validates and activates a Destination for the given source in the state object
func (state *State) ActivateRoute(sourceID string, destinationID string) (e error) {
	var route *Route

	for i, r := range state.Routes {
		if r.SourceID == sourceID {
			route = &state.Routes[i]
			break
		}
	}

	if route.SourceID == "" {
		return fmt.Errorf("Could not find Route with Source ID %v", sourceID)
	}

	if _, ok := state.Destinations[destinationID]; !ok {
		return fmt.Errorf("Could not find destination with ID %v", destinationID)
	}

	if _, ok := route.Destinations[destinationID]; !ok {
		return fmt.Errorf("Destination %v is not associated to route with source ID %v", destinationID, sourceID)
	}

	if route.Destinations[destinationID].Active {
		return fmt.Errorf("Destination %v is already active", destinationID)
	}

	route.Destinations[destinationID].Active = true

	return nil
}

// DeactivateRoute validates and deactivates a Destination for the given source in the state object
func (state *State) DeactivateRoute(sourceID string, destinationID string) (e error) {
	var route *Route

	for i, r := range state.Routes {
		if r.SourceID == sourceID {
			route = &state.Routes[i]
			break
		}
	}

	if route.SourceID == "" {
		return fmt.Errorf("Could not find Route with Source ID %v", sourceID)
	}

	if _, ok := state.Destinations[destinationID]; !ok {
		return fmt.Errorf("Could not find destination with ID %v", destinationID)
	}

	if _, ok := route.Destinations[destinationID]; !ok {
		return fmt.Errorf("Destination %v is not associated to route with source ID %v", destinationID, sourceID)
	}

	if !route.Destinations[destinationID].Active {
		return fmt.Errorf("Destination %v is already deactivated", destinationID)
	}

	route.Destinations[destinationID].Active = false

	return nil
}

// RemoveRoute validates and removes a Route from the state object
func (state *State) RemoveRoute(sourceID string) (e error) {
	var route *Route
	var routeIndex int

	for i, r := range state.Routes {
		if r.SourceID == sourceID {
			route = &state.Routes[i]
			routeIndex = i
			break
		}
	}

	if route == nil {
		return fmt.Errorf("Could not find Route with Source ID %v", sourceID)
	}

	if len(route.Destinations) > 0 {
		return fmt.Errorf("Could not remove route as it has %v destinations. Unlink destinations first", len(route.Destinations))
	}

	state.Routes = append(state.Routes[:routeIndex], state.Routes[routeIndex+1:]...)

	return nil
}
