import cdk = require('@aws-cdk/core');
import elb = require('@aws-cdk/aws-elasticloadbalancingv2');
import ec2 = require('@aws-cdk/aws-ec2');

export interface Props {
  cdk: cdk.StackProps,
  cidr: string,
  allowedCIDRs: AllowedCIDR[],
}

export interface AllowedCIDR{
  name: string,
  cidr: string,
}

export class FailsafeVPCStack extends cdk.Stack {
  VPC: ec2.Vpc

  constructor(scope: cdk.Construct, id: string, props: Props) {
    super(scope, id, props.cdk);

    this.VPC = new ec2.Vpc(this, 'FailsafeVPC', {
      maxAzs: 3,
      cidr: props.cidr,
    });

    // Side note, we can't do route tables (that I can see) with CDK yet, so that part is done manually

    const acl = ec2.NetworkAcl.fromNetworkAclId(this, 'FailsafeACL', this.VPC.vpcDefaultNetworkAcl)

    // Make sure that our production network can talk to the VPC
    acl.addEntry('PRODNETWORK_INGRESS', {
      ruleAction: ec2.Action.ALLOW,
      ruleNumber: 10,
      traffic: ec2.AclTraffic.allTraffic(),
      cidr: ec2.AclCidr.ipv4('10.0.0.0/8'),
      direction: ec2.TrafficDirection.INGRESS,
    })

    acl.addEntry('PRODNETWORK_EGRESS', {
      ruleAction: ec2.Action.ALLOW,
      ruleNumber: 10,
      traffic: ec2.AclTraffic.allTraffic(),
      cidr: ec2.AclCidr.ipv4('10.0.0.0/8'),
      direction: ec2.TrafficDirection.EGRESS,
    })

    // Iterate through the CIDRs we've been passed and add them to the ACL
    for (var i = 0; i < props.allowedCIDRs.length; i++){
      acl.addEntry(props.allowedCIDRs[i].name + '_INGRESS', {
        ruleAction: ec2.Action.ALLOW,
        ruleNumber: 100 + (10 * (i+1)),
        traffic: ec2.AclTraffic.tcpPort(1935),
        cidr: ec2.AclCidr.ipv4(props.allowedCIDRs[i].cidr),
        direction: ec2.TrafficDirection.INGRESS,
      })

      acl.addEntry(props.allowedCIDRs[i].name + '_EGRESS', {
        ruleAction: ec2.Action.ALLOW,
        ruleNumber: 100 + (10 * (i+1)),
        traffic: ec2.AclTraffic.allTraffic(),
        cidr: ec2.AclCidr.ipv4(props.allowedCIDRs[i].cidr),
        direction: ec2.TrafficDirection.EGRESS,
      })
    }
  }
}
