package main

import (
	awsBackend "code.justin.tv/event-engineering/goldengate/pkg/aws/backend"
	"code.justin.tv/event-engineering/goldengate/pkg/env-parameter-resolver"
	gg "code.justin.tv/event-engineering/goldengate/pkg/goldengate"
	"code.justin.tv/event-engineering/goldengate/pkg/handlers"
	jiraBackend "code.justin.tv/event-engineering/goldengate/pkg/jira/backend"
	"code.justin.tv/event-engineering/goldengate/pkg/oauth"
	pdBackend "code.justin.tv/event-engineering/goldengate/pkg/pagerduty/backend"
	"code.justin.tv/event-engineering/goldengate/pkg/ssm-parameter-resolver"
	twilioBackend "code.justin.tv/event-engineering/goldengate/pkg/twilio/backend"
	"github.com/Sirupsen/logrus"
	"github.com/aws/aws-lambda-go/lambda"
	"github.com/pkg/errors"
	"net/http"
	"time"
)

var handler handlers.Handlers

type backendResolver struct {
	gg.Parameters
	gg.SecureParameters
	logger     *logrus.Logger
	awsBackend awsBackend.Client
}

func (b *backendResolver) GetPagerdutyBackend() pdBackend.Client {
	return pdBackend.New(b.PagerDutyAPIKey, b.logger)
}

func (b *backendResolver) GetTwilioBackend() twilioBackend.Client {
	return twilioBackend.New(b.TwilioAccountSid, b.TwilioAuthToken, &http.Client{})
}

func (b *backendResolver) GetAWSBackend() (awsBackend.Client, error) {
	if b.awsBackend != nil {
		return b.awsBackend, nil
	}

	// There's no params to pass in just yet as we just use IAM roles, we might need to pass in things in future though
	backend, err := awsBackend.New()

	if err != nil {
		return nil, err
	}

	b.awsBackend = backend

	return b.awsBackend, nil
}

func (b *backendResolver) GetJiraBackend() (jiraBackend.Client, error) {
	oauthClient, err := oauth.GenerateOAuthClient(b.JiraPrivateKey, b.JiraConsumerKey, b.JiraToken, b.JiraTokenSecret)
	if err != nil {
		return nil, errors.Wrap(err, "Could not generate JIRA OAuth client")
	}

	client, err := jiraBackend.New(oauthClient, b.JiraHostname)

	if err != nil {
		return nil, errors.Wrap(err, "Could not generate JIRA client")
	}

	return client, nil
}

func init() {
	logger := logrus.New()
	logger.Level = logrus.DebugLevel
	logger.SetNoLock()

	paramResolver := envparameterresolver.New(logger)

	params, err := paramResolver.GetParameters()
	if err != nil {
		logger.Error(errors.Wrap(err, "Could not get parameters"))
		panic(err)
	}

	backendResolver := &backendResolver{
		Parameters: params,
	}

	// Resolve the AWS backend as we need AWS SSM to resolve the secure parameters (like access keys etc.)
	awsBackend, err := backendResolver.GetAWSBackend()
	if err != nil {
		logger.Error(errors.Wrap(err, "Could not get aws backend"))
		panic(err)
	}

	secureParamResolver := ssmparameterresolver.New(logger, awsBackend)

	secureParams, err := secureParamResolver.GetParameters()
	if err != nil {
		logger.Error(errors.Wrap(err, "Could not get secure parameters"))
		panic(err)
	}

	backendResolver.SecureParameters = secureParams

	teamUsersCacheTime := 10 * time.Minute

	handler, err = handlers.New(logger, params, secureParams, backendResolver, teamUsersCacheTime)

	if err != nil {
		panic(err)
	}
}

func main() {
	lambda.Start(handler.Handle)
}
