package aws

import (
	"code.justin.tv/event-engineering/goldengate/pkg/aws/backend"
	logging "code.justin.tv/event-engineering/goldengate/pkg/logging/backend"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3manager"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/aws/aws-sdk-go/service/ssm"
	"github.com/pkg/errors"
	"io"
	"time"
)

var presignedURLExpiry = 7 * 24 * time.Hour

// New generates a new AWS Client with the supplied backend client
func New(awsClient backend.Client, logger logging.Logger) *Client {
	return &Client{
		aws:    awsClient,
		logger: logger,
	}
}

// Client is an AWS client containing methods for use in the goldengate system
type Client struct {
	aws    backend.Client
	logger logging.Logger
}

// UploadRecording uploads the supplied data to the bucket / key and returns a presigned URL to download
func (c *Client) UploadRecording(data io.ReadCloser, bucketName, destinationKey string) (signedURL string, err error) {
	contentType := "audio/mpeg"

	_, err = c.aws.S3Upload(&s3manager.UploadInput{
		Bucket:      &bucketName,
		Key:         &destinationKey,
		ContentType: &contentType,
		Body:        data,
	})

	if err != nil {
		return "", errors.Wrap(err, "Failed to upload recording")
	}

	c.logger.Debug("Uploaded recording ", destinationKey)

	req, _ := c.aws.S3GetObjectRequest(&s3.GetObjectInput{
		Bucket: &bucketName,
		Key:    &destinationKey,
	})

	presignedURL, err := req.Presign(presignedURLExpiry)
	if err != nil {
		return "", errors.Wrap(err, "Failed to generate presigned URL")
	}

	return presignedURL, nil
}

// SendSQSMessage calls the underlying SendSQSMessage from the aws backend
func (c *Client) SendSQSMessage(input *sqs.SendMessageInput) (*sqs.SendMessageOutput, error) {
	return c.aws.SendSQSMessage(input)
}

// DDBGetItem calls the underlying DDBGetItem from the aws backend
func (c *Client) DDBGetItem(input *dynamodb.GetItemInput) (*dynamodb.GetItemOutput, error) {
	return c.aws.DDBGetItem(input)
}

// DDBPutItem calls the underlying DDBPutItem from the aws backend
func (c *Client) DDBPutItem(input *dynamodb.PutItemInput) (*dynamodb.PutItemOutput, error) {
	return c.aws.DDBPutItem(input)
}

// DDBDeleteItem calls the underlying DDBDeleteItem from the aws backend
func (c *Client) DDBDeleteItem(input *dynamodb.DeleteItemInput) (*dynamodb.DeleteItemOutput, error) {
	return c.aws.DDBDeleteItem(input)
}

// DDBQuery calls the underlying DDBQuery from the aws backend
func (c *Client) DDBQuery(input *dynamodb.QueryInput) (*dynamodb.QueryOutput, error) {
	return c.aws.DDBQuery(input)
}

// SSMGetParameters calls the underlying SSMGetParameters from the aws backend
func (c *Client) SSMGetParameters(input *ssm.GetParametersInput) (*ssm.GetParametersOutput, error) {
	return c.aws.SSMGetParameters(input)
}
