package backend

import (
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/ec2rolecreds"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3manager"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/aws/aws-sdk-go/service/ssm"
	"github.com/pkg/errors"
)

// Client contains methods for AWS interactions to allow easier data mocking for tests.
// If this interface changes, counterfeiter must be re-run using `make mocks`
type Client interface {
	S3Upload(input *s3manager.UploadInput, options ...func(*s3manager.Uploader)) (*s3manager.UploadOutput, error)
	S3GetObjectRequest(input *s3.GetObjectInput) (req *request.Request, output *s3.GetObjectOutput)
	SendSQSMessage(input *sqs.SendMessageInput) (*sqs.SendMessageOutput, error)
	DDBGetItem(input *dynamodb.GetItemInput) (*dynamodb.GetItemOutput, error)
	DDBPutItem(input *dynamodb.PutItemInput) (*dynamodb.PutItemOutput, error)
	DDBDeleteItem(input *dynamodb.DeleteItemInput) (*dynamodb.DeleteItemOutput, error)
	DDBQuery(input *dynamodb.QueryInput) (*dynamodb.QueryOutput, error)
	SSMGetParameters(input *ssm.GetParametersInput) (*ssm.GetParametersOutput, error)
}

type client struct {
	s3         *s3.S3
	s3Uploader *s3manager.Uploader
	sqs        *sqs.SQS
	ddb        *dynamodb.DynamoDB
	ssm        *ssm.SSM
}

// New calls the NewClient function from the base jira package
func New() (Client, error) {
	// Set up S3 Client
	creds := credentials.NewChainCredentials(
		[]credentials.Provider{
			&credentials.EnvProvider{},
			&ec2rolecreds.EC2RoleProvider{},
		})

	sess, err := session.NewSession(&aws.Config{
		Region:      aws.String("us-west-2"),
		Credentials: creds,
	})

	if err != nil {
		return nil, errors.Wrap(err, "Error creating AWS session")
	}

	return &client{
		s3:         s3.New(sess),
		s3Uploader: s3manager.NewUploader(sess),
		sqs:        sqs.New(sess),
		ddb:        dynamodb.New(sess),
		ssm:        ssm.New(sess),
	}, nil
}

// S3Upload calls the underlying S3Upload from the s3Uploader backend
func (c *client) S3Upload(input *s3manager.UploadInput, options ...func(*s3manager.Uploader)) (*s3manager.UploadOutput, error) {
	return c.s3Uploader.Upload(input, options...)
}

// S3GetObjectRequest calls the underlying GetObjectRequest from the s3 backend
func (c *client) S3GetObjectRequest(input *s3.GetObjectInput) (req *request.Request, output *s3.GetObjectOutput) {
	return c.s3.GetObjectRequest(input)
}

// SendSQSMessage calls the underlying SendMessage from the sqs backend
func (c *client) SendSQSMessage(input *sqs.SendMessageInput) (*sqs.SendMessageOutput, error) {
	return c.sqs.SendMessage(input)
}

// DDBGetItem calls the underlying GetItem from the dynamodb backend
func (c *client) DDBGetItem(input *dynamodb.GetItemInput) (*dynamodb.GetItemOutput, error) {
	return c.ddb.GetItem(input)
}

// DDBPutItem calls the underlying PutItem from the dynamodb backend
func (c *client) DDBPutItem(input *dynamodb.PutItemInput) (*dynamodb.PutItemOutput, error) {
	return c.ddb.PutItem(input)
}

// DDBDeleteItem calls the underlying DeleteItem from the dynamodb backend
func (c *client) DDBDeleteItem(input *dynamodb.DeleteItemInput) (*dynamodb.DeleteItemOutput, error) {
	return c.ddb.DeleteItem(input)
}

// DDBQuery calls the underlying Query from the dynamodb backend
func (c *client) DDBQuery(input *dynamodb.QueryInput) (*dynamodb.QueryOutput, error) {
	return c.ddb.Query(input)
}

// SSMGetParameters calls the underlying GetParameters from the ssm backend
func (c *client) SSMGetParameters(input *ssm.GetParametersInput) (*ssm.GetParametersOutput, error) {
	return c.ssm.GetParameters(input)
}
