package goldengate

import (
	"reflect"
	"strconv"
	"time"
)

// SecureParameters contains all the parameters that are stored securely required for goldengate
type SecureParameters struct {
	TwilioAccountSid              string `ssm:"/goldengate/twilioAccountSid" env:"twilioAccountSid"`
	TwilioAuthToken               string `ssm:"/goldengate/twilioAuthToken" env:"twilioAuthToken"`
	PagerDutyAPIKey               string `ssm:"/goldengate/pagerDutyApiKey" env:"pagerDutyApiKey"`
	PagerdutyTwilioIntegrationKey string `ssm:"/goldengate/pdTwilioIntegrationKey" env:"pdTwilioIntegrationKey"`
	JiraToken                     string `ssm:"/goldengate/jiraToken" env:"jiraToken"`
	JiraTokenSecret               string `ssm:"/goldengate/jiraTokenSecret" env:"jiraTokenSecret"`
	JiraPrivateKey                string `ssm:"/goldengate/jiraPrivateKey" env:"jiraPrivateKey"`
}

// Parameters contains all the non-secure parameters that are required for goldengate
type Parameters struct {
	TwilioBridgeNumber  string `env:"twilioBridgeNumber"`
	ConferenceName      string `env:"conferenceName"`
	RecordingBucketName string `env:"recordingBucketName"`
	HoldMusicURL        string `env:"holdMusicURL"`
	JiraHostname        string `env:"jiraHostname"`
	JiraConsumerKey     string `env:"jiraConsumerKey"`
	EscalationPolicyID  string `env:"escalationPolicyID"`
	TeamID              string `env:"teamID"`
	BlockedNumbers      string `env:"blockedNumbers"`
	BlockedPrefixes     string `env:"blockedPrefixes"`
	SQSQueueURL         string `env:"sqsQueueURL"`
	DynamoDBTableName   string `env:"dynamoDbTableName"`
	JiraEnabled         bool   `env:"jiraEnabled"`
	PDEnabled           bool   `env:"pdEnabled"`
	BatphoneNumbers     string `env:"batphoneNumbers"`
	TeamUsersCacheTime  time.Duration
}

// ParameterResolver resolves parameters returning a Parameters object
type ParameterResolver interface {
	GetParameters() (Parameters, error)
}

// SecureParameterResolver resolves secure parameters returning a SecureParameters object
type SecureParameterResolver interface {
	GetParameters() (SecureParameters, error)
}

// MapParameters will map the values from the parameter mapping to the target struct
func MapParameters(values map[string]*ParamMapping, target interface{}) {
	paramType := reflect.TypeOf(target)

	if paramType.Kind() == reflect.Ptr {
		paramType = paramType.Elem()
	}

	s := reflect.ValueOf(target)

	if s.Kind() == reflect.Ptr {
		s = s.Elem()
	}

	if s.Kind() != reflect.Struct {
		return
	}

	for i := 0; i < paramType.NumField(); i++ {
		field := paramType.Field(i)
		if paramMapping, ok := values[field.Name]; ok {
			f := s.Field(i)
			if f.IsValid() && f.CanSet() {
				switch f.Kind() {
				case reflect.String:
					f.SetString(paramMapping.Value)
				case reflect.Int:
					x, err := strconv.ParseInt(paramMapping.Value, 10, 64)
					if err == nil && !f.OverflowInt(x) {
						f.SetInt(x)
					}
				case reflect.Bool:
					x, err := strconv.ParseBool(paramMapping.Value)
					if err == nil {
						f.SetBool(x)
					}
				}
			}
		}
	}
}

// GetParameters will return the mapping for the supplied tag name on the struct passed in paramsFor
func GetParameters(paramsFor interface{}, tagName string) map[string]*ParamMapping {
	params := make(map[string]*ParamMapping)
	paramType := reflect.TypeOf(paramsFor)

	if paramType.Kind() == reflect.Ptr {
		paramType = paramType.Elem()
	}

	if paramType.Kind() != reflect.Struct {
		return nil
	}

	for i := 0; i < paramType.NumField(); i++ {
		field := paramType.Field(i)
		if paramName, ok := field.Tag.Lookup(tagName); ok {
			params[field.Name] = &ParamMapping{TagValue: paramName}
		}
	}

	return params
}

// ParamMapping represents a mapping between a tag and a value
type ParamMapping struct {
	TagValue string
	Value    string
}
