package handlers

import (
	"context"
	"fmt"
	goTwilio "github.com/kevinburke/twilio-go"
	"github.com/pkg/errors"
	"sync"
)

func (h *handlers) getOnCallTelephoneNumbers(escalationPolicyID string, escalationLevel int) (map[string]string, error) {
	users, err := h.pd.GetOncallUsers(escalationPolicyID, escalationLevel)
	if err != nil {
		return nil, err
	}

	if len(users) == 0 {
		return nil, errors.New("No oncall users with registered numbers could be found")
	}

	h.logger.Debugf("Retrieved %v On Call users", len(users))

	// Map the user IDs into a simple slice so that we can retrieve the full user objects to get the phone numbers
	userIDs := make([]string, 0)
	for _, user := range users {
		userIDs = append(userIDs, user.ID)
	}

	return h.getTeamUserNumbers(userIDs), nil
}

func (h *handlers) getTeamUserNumbers(userIDs []string) map[string]string {
	teamUsers := h.pd.GetTeamUsers(h.TeamID, userIDs, h.TeamUsersCacheTime)

	// Now that we have a list of full user objects populate a simple list of telephone numbers to dial
	numbers := make(map[string]string)
	for _, tu := range teamUsers {
		numbers[tu.ID] = getUserNumber(tu)
	}

	return numbers
}

func (h *handlers) dialOnCalls(escalationLevel int, pickupURL string) ([]*string, error) {
	// Get the numbers for the oncall users at escalation level 2
	numbers, err := h.getOnCallTelephoneNumbers(h.EscalationPolicyID, escalationLevel)

	if err != nil {
		return nil, errors.Wrap(err, "Error getting oncall telephone numbers")
	}

	if len(numbers) == 0 {
		return nil, fmt.Errorf("Zero oncall numbers could be found at escalation level %v", escalationLevel)
	}

	successfulDials := 0
	outgoingCalls := make([]*string, 0)
	for userID, number := range numbers {
		if number == "" {
			h.logger.Debugf("Skipping user %v as it has no telephone number", userID)
			continue
		}

		callSid, dialErr := h.twilio.DialNumber(h.TwilioBridgeNumber, number, pickupURL)
		if dialErr != nil {
			h.logger.Errorf("Error dialling oncall user %v - %v", userID, dialErr)
		} else {
			h.logger.Debugf("Successfully dialled oncall user %v", userID)
			successfulDials++
			outgoingCalls = append(outgoingCalls, &callSid)
		}
	}

	if successfulDials == 0 {
		return nil, errors.New("Zero successful dials")
	}

	return outgoingCalls, nil
}

func (h *handlers) getCalls(ctx context.Context, callSids []*string) (calls []*goTwilio.Call) {
	calls = make([]*goTwilio.Call, 0)
	var wg sync.WaitGroup
	var listLock sync.Mutex

	for _, sid := range callSids {
		wg.Add(1)
		go func(callSid *string) {
			defer wg.Done()
			call, err := h.twilio.GetCall(ctx, *callSid)
			if err != nil {
				h.logger.Error("Error retrieving call details ", err)
				return
			}
			listLock.Lock()
			calls = append(calls, call)
			listLock.Unlock()
		}(sid)
	}

	wg.Wait()

	return
}

func (h *handlers) hangupOutgoingCalls(ctx context.Context, outgoingCallSids []*string) {
	teamCalls := h.getCalls(ctx, outgoingCallSids)
	// Hang up all other outgoing calls to the team that may exist
	for _, call := range teamCalls {
		if call.Status == "ringing" {
			_, huErr := h.twilio.Hangup(call.Sid)
			if huErr != nil {
				h.logger.Warnf("Error hanging up call %v - %v", call.Sid, huErr)
			}
		}
	}
}
