package handlers

import (
	pd "code.justin.tv/event-engineering/goldengate/pkg/pagerduty/backend"
	"context"
	"errors"
	goTwilio "github.com/kevinburke/twilio-go"
	"github.com/stretchr/testify/assert"
	"strings"
	"testing"
)

func TestGetOnCallTelephoneNumbers(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	// Test erroring ListOnCalls call
	backend.pd.ListOnCallsStub = func(escalationPolicyId string) (*pd.ListOnCallsResponse, error) {
		return nil, errors.New("test")
	}

	_, err = backend.handlersPriv.getOnCallTelephoneNumbers("EP1", 1)
	a.NotNil(err)
	a.True(strings.HasPrefix(err.Error(), "Failed to get Oncall Users:"), "error message should start with Failed to get Oncall Users: - got `%v`", err.Error())

	// Test empty users response
	backend.pd.ListOnCallsStub = func(escalationPolicyId string) (*pd.ListOnCallsResponse, error) {
		return &pd.ListOnCallsResponse{
			OnCalls: make([]*pd.OnCall, 0),
		}, nil
	}

	_, err = backend.handlersPriv.getOnCallTelephoneNumbers("EP1", 1)
	a.NotNil(err)
	a.EqualValues("No Oncall Users returned", err.Error())

	// Test no users with phone number response
	backend.pd.ListOnCallsStub = func(escalationPolicyId string) (*pd.ListOnCallsResponse, error) {
		return &pd.ListOnCallsResponse{
			OnCalls: []*pd.OnCall{
				&pd.OnCall{
					EscalationLevel: 1,
					User: &pd.OnCallUser{
						ID: "User1",
						ContactMethods: []*pd.ContactMethodReference{
							&pd.ContactMethodReference{
								ID:   "CM1",
								Type: "email_contact_method_reference",
							},
						},
					},
				},
			},
		}, nil
	}

	_, err = backend.handlersPriv.getOnCallTelephoneNumbers("EP1", 1)
	a.NotNil(err)
	a.EqualValues("No oncall users with registered numbers could be found", err.Error())

	// Test valid oncall user, but not in team
	backend.pd.ListOnCallsStub = func(escalationPolicyId string) (*pd.ListOnCallsResponse, error) {
		return &pd.ListOnCallsResponse{
			OnCalls: []*pd.OnCall{
				&pd.OnCall{
					EscalationLevel: 1,
					User: &pd.OnCallUser{
						ID: "USER1",
						ContactMethods: []*pd.ContactMethodReference{
							&pd.ContactMethodReference{
								ID:   "CM1",
								Type: "phone_contact_method_reference",
							},
						},
					},
				},
			},
		}, nil
	}

	resp, err := backend.handlersPriv.getOnCallTelephoneNumbers("EP1", 1)
	a.Nil(err)
	a.EqualValues(0, len(resp))

	// Test valid oncall user and in team list
	backend.pd.ListTeamUsersStub = func(teamID string) (*pd.ListUsersResponse, error) {
		return &pd.ListUsersResponse{
			Users: []*pd.User{
				&pd.User{
					ID: "USER1",
					ContactMethods: []*pd.ContactMethod{
						&pd.ContactMethod{
							ID:          "CM1",
							Type:        "phone_contact_method",
							CountryCode: 44,
							Address:     "1234567890",
						},
					},
				},
			},
		}, nil
	}

	resp, err = backend.handlersPriv.getOnCallTelephoneNumbers("EP1", 1)
	a.Nil(err)
	a.EqualValues(1, len(resp))
	a.EqualValues("+441234567890", resp["USER1"])
}

func TestGetCalls(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	backend.twilio.GetCallStub = getCallSuccess
	callSid := "test_call_sid"
	calls := backend.handlersPriv.getCalls(context.TODO(), []*string{&callSid})

	a.EqualValues(1, len(calls))

	// Test erroring GetCall function
	backend.twilio.GetCallStub = func(ctx context.Context, sid string) (*goTwilio.Call, error) {
		return nil, errors.New("err making call")
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Error retrieving call details ", args[0])
		a.EqualValues("err making call", error(args[1].(error)).Error())
	}

	calls = backend.handlersPriv.getCalls(context.TODO(), []*string{&callSid})
	a.EqualValues(0, len(calls))
	a.EqualValues(1, backend.logger.ErrorCallCount())
}

func TestHangup(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	backend.twilio.GetCallStub = func(ctx context.Context, sid string) (*goTwilio.Call, error) {
		return &goTwilio.Call{
			Sid:    "ABC123",
			Status: "ringing",
			From:   "+441234567890",
			To:     "+1975638394",
		}, nil
	}

	callSid := "test_call_sid"
	backend.handlersPriv.hangupOutgoingCalls(context.TODO(), []*string{&callSid})
	a.EqualValues(0, backend.logger.WarnfCallCount())

	// Test erroring Hangup function
	backend.twilio.HangupStub = func(sid string) (*goTwilio.Call, error) {
		return nil, errors.New("err hanging up call")
	}

	backend.logger.WarnfStub = func(format string, args ...interface{}) {
		a.EqualValues("Error hanging up call %v - %v", format)
		a.EqualValues("ABC123", args[0])
		a.EqualValues("err hanging up call", error(args[1].(error)).Error())
	}

	backend.handlersPriv.hangupOutgoingCalls(context.TODO(), []*string{&callSid})
	a.EqualValues(1, backend.logger.WarnfCallCount())
}
