package handlers

import (
	"context"
	"github.com/aws/aws-lambda-go/events"
	"github.com/pkg/errors"
	"net/url"
)

func (h *handlers) onConferenceEvent(ctx context.Context, request events.APIGatewayProxyRequest, urlParams url.Values) (events.APIGatewayProxyResponse, error) {
	h.logger.Debug("ConferenceEvent fired", request.Body, request.QueryStringParameters)

	switch urlParams.Get("StatusCallbackEvent") {
	case "participant-join":
		return h.onJoinConference(ctx, urlParams)
	case "conference-end":
		return h.onEndConference(ctx, urlParams)
	}

	return events.APIGatewayProxyResponse{}, errors.New("Not Implemented")
}

func (h handlers) onJoinConference(ctx context.Context, data url.Values) (events.APIGatewayProxyResponse, error) {
	// Grab the telephone number from the call info
	call, err := h.twilio.GetCall(ctx, data.Get("CallSid"))
	if err != nil {
		return er, err
	}

	// Get the state for this conference based on the initial call sid
	state, err := h.getStateFromCall(data.Get("InitialCallSid"))
	if err != nil {
		h.logger.Errorf("Could not get state from initial call sid %v - %v", data.Get("InitialCallSid"), err)
	} else if state.ConferenceSid == "" {
		// If we've not linked the initial call and the conference together yet in the state, then do so
		state.ConferenceSid = data.Get("ConferenceSid")
		err := h.putState(state)
		if err != nil {
			h.logger.Error("Could not update conference sid ", err)
		}
	}

	// Grab all the phone numbers for the event engineering team and check to see if the number matches to whoever just entered the conference
	numbers := h.getTeamUserNumbers(nil)

	if len(numbers) == 0 {
		return er, errors.New("Zero team numbers could be found")
	}

	// Could be from or to here, From if they dialled in manually or To if they picked up the outbound call
	if isInMap(string(call.To), numbers) || isInMap(string(call.From), numbers) {
		h.logger.Debug("Acking Pagerduty incident")
		ackErr := h.pd.AckEvent(h.PagerdutyTwilioIntegrationKey, h.ConferenceName, "Incident Acknowledged")
		if ackErr != nil {
			h.logger.Error("Error acknowledging Pagerduty Incident: ", ackErr)
		}

		// Grab the outgoing calls to the team that might still be ringing and hangup
		// Also update the state so that we know an EE member has answered the call of duty
		if state != nil {
			state.TeamPickedUp = true
			err := h.putState(state)
			if err != nil {
				h.logger.Error("Could not mark TeamPickedUp ", err)
			}

			callSids := make([]*string, 0)
			for _, callSid := range state.OutgoingCalls {
				// Don't hang up if it's the call that triggered this event, we check that any outgoing calls are still ringing before hanging up in
				// h.hangupOutgoingCalls but this is just to be sure we don't cut off this call accidentally if something failed to update in time
				if *callSid != data.Get("CallSid") {
					callSids = append(callSids, callSid)
				}
			}
			h.hangupOutgoingCalls(ctx, callSids)
		}
	}

	return er, nil
}

func (h handlers) onEndConference(ctx context.Context, data url.Values) (events.APIGatewayProxyResponse, error) {
	// Resolve the pagerduty incident
	h.logger.Debug("Resolving Pagerduty incident")
	err := h.pd.ResolveEvent(h.PagerdutyTwilioIntegrationKey, h.ConferenceName, "Incident Resolved")
	if err != nil {
		h.logger.Error("Error resolving Pagerduty Incident: ", err)
	}

	// Grab the state
	state, err := h.getStateFromCall(data.Get("InitialCallSid"))
	if err != nil {
		h.logger.Errorf("Could not get state from initial call sid %v - %v", data.Get("InitialCallSid"), err)
	} else {
		// Hang up all remaining outgoing calls that are still rining (hopefully none)
		h.hangupOutgoingCalls(ctx, state.OutgoingCalls)

		// If this conference has no recordings then delete the state as we no longer need it, however leave it present if there are recordings
		// the state will be cleaned up by the recording event handler when the recording is ready to be uploaded to S3
		recordings, err := h.twilio.GetConferenceRecordings(ctx, data.Get("ConferenceSid"))
		if err != nil {
			h.logger.Errorf("Error retrieving conference recordings %v - %v ", data.Get("ConferenceSid"), err)
		} else if len(recordings.Recordings) == 0 {
			h.logger.Debugf("No recordings for conference %v deleting state", data.Get("ConferenceSid"))
			err := h.deleteState(state)
			if err != nil {
				h.logger.Error("Error deleting state ", err)
			}
		}
	}

	return er, nil
}
