package handlers

import (
	twilioBackend "code.justin.tv/event-engineering/goldengate/pkg/twilio/backend"
	"context"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	goTwilio "github.com/kevinburke/twilio-go"
	pdEvents "github.com/marcw/pagerduty"
	"github.com/pkg/errors"
	"github.com/stretchr/testify/assert"
	"testing"
)

var joinRequest = Request{
	APIGatewayProxyRequest: events.APIGatewayProxyRequest{
		Path: "/conferenceevent",
		Body: "StatusCallbackEvent=participant-join&CallSid=conf_event_call_sid&InitialCallSid=initial_call_sid&ConferenceSid=conference_sid",
	},
}

var getTeamCallSuccess = func(ctx context.Context, sid string) (*goTwilio.Call, error) {
	return &goTwilio.Call{
		Sid:  "ABC123",
		From: "+441234567890",
		To:   "+1975638394",
	}, nil
}

var getConferenceRecordings = func(ctx context.Context, conferenceSid string) (*twilioBackend.ConferenceRecordingsResponse, error) {
	return &twilioBackend.ConferenceRecordingsResponse{}, nil
}

func TestUnhandledCallback(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test not implemented status callback
	request := Request{
		APIGatewayProxyRequest: events.APIGatewayProxyRequest{
			Path: "/conferenceevent",
			Body: "StatusCallbackEvent=not-handled",
		},
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Error executing handler: ", args[0])
		a.EqualValues("Not Implemented", error(args[1].(error)).Error())
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorCallCount())
}

func TestGetCallError(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test erroing GetCall
	request := joinRequest

	backend.twilio.GetCallStub = func(ctx context.Context, sid string) (*goTwilio.Call, error) {
		// These IDs are declared in handlers_test.go in the createMockBackend function
		a.EqualValues("conf_event_call_sid", sid)
		return nil, errors.New("getcall error")
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Error executing handler: ", args[0])
		a.EqualValues("getcall error", error(args[1].(error)).Error())
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorCallCount())
}

func TestGetStateError(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test no team users returned
	request := joinRequest
	backend.twilio.GetCallStub = getCallSuccess

	backend.aws.DDBGetItemStub = func(input *dynamodb.GetItemInput) (*dynamodb.GetItemOutput, error) {
		return nil, errors.New("Error getting state")
	}

	backend.logger.ErrorfStub = func(format string, args ...interface{}) {
		a.EqualValues("Could not get state from initial call sid %v - %v", format)
		a.EqualValues("initial_call_sid", args[0])
		a.EqualValues("Error getting state", error(args[1].(error)).Error())
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorfCallCount())
}

func TestPutStateError(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test put state error
	request := joinRequest
	backend.twilio.GetCallStub = getCallSuccess
	backend.aws.DDBGetItemStub = getStateSuccess
	backend.pd.ListTeamUsersStub = teamUsers
	backend.pd.SubmitEventStub = func(event *pdEvents.Event) (response *pdEvents.Response, statusCode int, err error) {
		return nil, 200, nil
	}

	backend.aws.DDBPutItemStub = func(input *dynamodb.PutItemInput) (*dynamodb.PutItemOutput, error) {
		return nil, errors.New("I CAN'T EVEN")
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Could not update conference sid ", args[0])
		a.EqualValues("I CAN'T EVEN", error(args[1].(error)).Error())
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorCallCount())
}

func TestNoTeamUsers(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test no team users returned
	request := joinRequest
	backend.twilio.GetCallStub = getCallSuccess
	backend.aws.DDBGetItemStub = getStateSuccess

	backend.aws.DDBGetItemStub = func(input *dynamodb.GetItemInput) (*dynamodb.GetItemOutput, error) {
		return &dynamodb.GetItemOutput{
			//Item: map[string]*dynamodb.AttributeValue{},
		}, nil
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Error executing handler: ", args[0])
		a.EqualValues("Zero team numbers could be found", error(args[1].(error)).Error())
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorCallCount())
}

func TestAckError(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test erroring AckEvent call
	request := joinRequest
	backend.twilio.GetCallStub = getTeamCallSuccess
	backend.aws.DDBGetItemStub = getStateSuccess
	backend.pd.ListTeamUsersStub = teamUsers

	backend.pd.SubmitEventStub = func(event *pdEvents.Event) (response *pdEvents.Response, statusCode int, err error) {
		return nil, 0, errors.New("ack error")
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Error acknowledging Pagerduty Incident: ", args[0])
		a.EqualValues("Failed to submit PD event: ack error", error(args[1].(error)).Error())
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorCallCount())
}

func TestSuccessfulJoinEvent(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test successful AckEvent call
	request := joinRequest
	backend.twilio.GetCallStub = getCallSuccess
	backend.aws.DDBGetItemStub = getStateSuccess
	backend.pd.ListTeamUsersStub = teamUsers

	backend.pd.SubmitEventStub = func(event *pdEvents.Event) (response *pdEvents.Response, statusCode int, err error) {
		return nil, 200, nil
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(0, backend.logger.ErrorCallCount())
}

func TestTeamPickedUpStateError(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test successful AckEvent call
	request := joinRequest
	backend.twilio.GetCallStub = getTeamCallSuccess
	backend.pd.ListTeamUsersStub = teamUsers

	backend.pd.SubmitEventStub = func(event *pdEvents.Event) (response *pdEvents.Response, statusCode int, err error) {
		return nil, 200, nil
	}

	backend.aws.DDBGetItemStub = func(input *dynamodb.GetItemInput) (*dynamodb.GetItemOutput, error) {
		callSid := "call_sid"
		conferenceSid := "conference_sid"
		return &dynamodb.GetItemOutput{
			Item: map[string]*dynamodb.AttributeValue{
				"call_sid":       &dynamodb.AttributeValue{S: &callSid},
				"conference_sid": &dynamodb.AttributeValue{S: &conferenceSid},
			},
		}, nil
	}

	backend.aws.DDBPutItemStub = func(input *dynamodb.PutItemInput) (*dynamodb.PutItemOutput, error) {
		return nil, errors.New("I CAN'T EVEN")
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Could not mark TeamPickedUp ", args[0])
		a.EqualValues("I CAN'T EVEN", error(args[1].(error)).Error())
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorCallCount())
}

func TestSuccessfulJoinEventWithTeamPickup(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test successful AckEvent call
	request := joinRequest
	backend.twilio.GetCallStub = func(ctx context.Context, sid string) (*goTwilio.Call, error) {
		return &goTwilio.Call{
			Sid:    "ABC123",
			Status: "ringing",
			From:   "+441234567890",
			To:     "+1975638394",
		}, nil
	}
	backend.aws.DDBGetItemStub = getStateSuccess
	backend.pd.ListTeamUsersStub = teamUsers

	backend.pd.SubmitEventStub = func(event *pdEvents.Event) (response *pdEvents.Response, statusCode int, err error) {
		return nil, 200, nil
	}

	backend.aws.DDBGetItemStub = func(input *dynamodb.GetItemInput) (*dynamodb.GetItemOutput, error) {
		callSid := "call_sid"
		conferenceSid := "conference_sid"
		outgoingCallSid := "outgoing_call_sid_1"
		outgoingCalls := []*string{&outgoingCallSid}

		return &dynamodb.GetItemOutput{
			Item: map[string]*dynamodb.AttributeValue{
				"call_sid":       &dynamodb.AttributeValue{S: &callSid},
				"conference_sid": &dynamodb.AttributeValue{S: &conferenceSid},
				"outgoing_calls": &dynamodb.AttributeValue{SS: outgoingCalls},
			},
		}, nil
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(0, backend.logger.ErrorCallCount())
	a.EqualValues(2, backend.twilio.GetCallCallCount())
	a.EqualValues(1, backend.twilio.HangupCallCount())
}

func TestEndEvent(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	request := Request{
		APIGatewayProxyRequest: events.APIGatewayProxyRequest{
			Path: "/conferenceevent",
			Body: "StatusCallbackEvent=conference-end",
		},
	}

	backend.aws.DDBGetItemStub = getStateSuccess
	backend.twilio.GetConferenceRecordingsStub = func(ctx context.Context, conferenceSid string) (*twilioBackend.ConferenceRecordingsResponse, error) {
		return nil, errors.New("reordings are busted")
	}

	// Test erroring ResolveEvent call
	backend.pd.SubmitEventStub = func(event *pdEvents.Event) (response *pdEvents.Response, statusCode int, err error) {
		return nil, 0, errors.New("resolve error")
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Error resolving Pagerduty Incident: ", args[0])
		a.EqualValues("Failed to submit PD event: resolve error", error(args[1].(error)).Error())
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorCallCount())

	// Test successful ResolveEvent call
	backend.pd.SubmitEventStub = func(event *pdEvents.Event) (response *pdEvents.Response, statusCode int, err error) {
		return nil, 200, nil
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	// ErrorCallCount should still only be 41
	a.EqualValues(1, backend.logger.ErrorCallCount())
}

func TestEndEventGetStateError(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	request := Request{
		APIGatewayProxyRequest: events.APIGatewayProxyRequest{
			Path: "/conferenceevent",
			QueryStringParameters: map[string]string{
				"InitialCallSid": "initial_call_sid",
			},
			Body: "StatusCallbackEvent=conference-end",
		},
	}

	backend.aws.DDBGetItemStub = func(input *dynamodb.GetItemInput) (*dynamodb.GetItemOutput, error) {
		return nil, errors.New("failed to get state because of dynamo")
	}

	backend.pd.SubmitEventStub = func(event *pdEvents.Event) (response *pdEvents.Response, statusCode int, err error) {
		return nil, 200, nil
	}

	backend.logger.ErrorfStub = func(format string, args ...interface{}) {
		a.EqualValues("Could not get state from initial call sid %v - %v", format)
		a.EqualValues("initial_call_sid", args[0])
		a.EqualValues("failed to get state because of dynamo", error(args[1].(error)).Error())
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorfCallCount())
}

func TestEndEventDeleteStateError(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	request := Request{
		APIGatewayProxyRequest: events.APIGatewayProxyRequest{
			Path: "/conferenceevent",
			QueryStringParameters: map[string]string{
				"InitialCallSid": "initial_call_sid",
			},
			Body: "StatusCallbackEvent=conference-end",
		},
	}
	backend.aws.DDBGetItemStub = getStateSuccess
	backend.twilio.GetConferenceRecordingsStub = getConferenceRecordings

	backend.aws.DDBDeleteItemStub = func(input *dynamodb.DeleteItemInput) (*dynamodb.DeleteItemOutput, error) {
		return nil, errors.New("failed to delete state because of dynamo")
	}

	backend.pd.SubmitEventStub = func(event *pdEvents.Event) (response *pdEvents.Response, statusCode int, err error) {
		return nil, 200, nil
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Error deleting state ", args[0])
		a.EqualValues("failed to delete state because of dynamo", error(args[1].(error)).Error())
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorCallCount())
}

func TestEndEventSuccess(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	request := Request{
		APIGatewayProxyRequest: events.APIGatewayProxyRequest{
			Path: "/conferenceevent",
			QueryStringParameters: map[string]string{
				"InitialCallSid": "initial_call_sid",
			},
			Body: "StatusCallbackEvent=conference-end",
		},
	}
	backend.aws.DDBGetItemStub = getEscalatedStateSuccess
	backend.twilio.GetConferenceRecordingsStub = getConferenceRecordings
	backend.pd.SubmitEventStub = func(event *pdEvents.Event) (response *pdEvents.Response, statusCode int, err error) {
		return nil, 200, nil
	}
	backend.twilio.GetCallStub = func(ctx context.Context, sid string) (*goTwilio.Call, error) {
		return &goTwilio.Call{
			Sid:    "ABC123",
			Status: "ringing",
			From:   "+441234567890",
			To:     "+1975638394",
		}, nil
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(0, backend.logger.ErrorCallCount())
	a.EqualValues(1, backend.twilio.GetCallCallCount())
	a.EqualValues(1, backend.twilio.HangupCallCount())
	a.EqualValues(1, backend.aws.DDBDeleteItemCallCount())
}
