package handlers

import (
	"code.justin.tv/event-engineering/goldengate/pkg/aws"
	"code.justin.tv/event-engineering/goldengate/pkg/goldengate"
	"code.justin.tv/event-engineering/goldengate/pkg/jira"
	logging "code.justin.tv/event-engineering/goldengate/pkg/logging/backend"
	"code.justin.tv/event-engineering/goldengate/pkg/pagerduty"
	"code.justin.tv/event-engineering/goldengate/pkg/twilio"
	"context"
	"fmt"
	"github.com/aws/aws-lambda-go/events"
	"github.com/pkg/errors"
	"net/url"
	"strings"
	"time"
)

// Request is a merged request object including all the properties from the events that we care about
type Request struct {
	events.APIGatewayProxyRequest
	events.CloudWatchEvent
	events.SQSEvent
}

// Response is a merged response object including all the properties from the responses we care about
type Response struct {
	events.APIGatewayProxyResponse
}

// Handlers is an object that can be used to handle requests
type Handlers interface {
	// Handle handles API gateway proxy requests, processing them and returning as appropriate
	Handle(ctx context.Context, request Request) (Response, error)
}

type handlers struct {
	logger          logging.Logger
	apiMappings     map[string]func(ctx context.Context, request events.APIGatewayProxyRequest, urlParams url.Values) (events.APIGatewayProxyResponse, error)
	jira            *jira.Client
	twilio          *twilio.Client
	pd              *pagerduty.Client
	aws             *aws.Client
	blockedNumbers  []string
	blockedPrefixes []string
	batphoneNumbers []string
	goldengate.Parameters
	goldengate.SecureParameters
}

// New generates a new handlers instance and registers all available callbacks
func New(logger logging.Logger, params goldengate.Parameters, secureParams goldengate.SecureParameters, backendResolver goldengate.BackendResolver, teamUsersCacheTime time.Duration) (Handlers, error) {
	return newHandlers(logger, params, secureParams, backendResolver, teamUsersCacheTime)
}

func newHandlers(logger logging.Logger, params goldengate.Parameters, secureParams goldengate.SecureParameters, backendResolver goldengate.BackendResolver, teamUsersCacheTime time.Duration) (*handlers, error) {
	h := &handlers{
		logger:      logger,
		apiMappings: make(map[string]func(ctx context.Context, request events.APIGatewayProxyRequest, urlParams url.Values) (events.APIGatewayProxyResponse, error)),
	}

	// Populate parameters
	h.Parameters = params
	h.SecureParameters = secureParams

	h.Parameters.TeamUsersCacheTime = teamUsersCacheTime

	// Parse the comma separated list of numbers into a slice
	h.blockedNumbers = strings.Split(h.BlockedNumbers, ",")
	h.blockedPrefixes = strings.Split(h.BlockedPrefixes, ",")
	h.batphoneNumbers = strings.Split(h.BatphoneNumbers, ",")

	// Set up mappings for API Gateway requests

	// /new is called when a new incoming call happens
	h.apiMappings["/new"] = h.onNewCall

	// /pickup happens when an outgoing call is picked up
	h.apiMappings["/pickup"] = h.onPickup

	// /conferenceevent happens when there is an update to the conference
	h.apiMappings["/conferenceevent"] = h.onConferenceEvent

	// /recordingevent happens when there is an update to the recording
	h.apiMappings["/recordingevent"] = h.onRecordingEvent

	// Set up Jira Client
	jiraClient, err := backendResolver.GetJiraBackend()
	if err != nil {
		return nil, err
	}
	h.jira = jira.New(jiraClient, logger)

	// Set up Twilio Client
	h.twilio = twilio.New(backendResolver.GetTwilioBackend(), logger)

	// Set up PagerDuty client
	h.pd = pagerduty.New(backendResolver.GetPagerdutyBackend(), logger)

	// Set up AWS clients
	awsClient, err := backendResolver.GetAWSBackend()
	if err != nil {
		return nil, err
	}
	h.aws = aws.New(awsClient, logger)

	return h, nil
}

func (h *handlers) Handle(ctx context.Context, request Request) (Response, error) {

	/*
		   We're doing something pretty funky here, we want the lambda execution context to be the same for the SQS trigger, Cloudwatch "cron" trigger and the API gateway trigger
			 Everything here is stateless and this is purely for efficiency, since it costs time to spin up new execution contexts and we expect under normal operation for this
			 function to be called in serial in most cases, my plan is to use Cloudwatch to keep the execution context fresh so that calls to the API respond faster,
			 this API gets called very rarely but we want it to be as fast to respond as possible so I'm hoping that this will result in always having a fresh execution context available
			 SQS triggers will be used when there is an ongoing call using a delayed queue to essentially operate like a setInterval() call to monitor the participants status
			 and time of the call in order to escalate the call if required, I could do this with Cloudwatch events but the minimum resolution is 1 minute and I need something
			 more in the region of 5 seconds. So... we have a Request object that is a merger of all the request objects that we're expecting, then we're testing the deserialised
			 values that indicate which type of request this is before shipping off that event to the appropriate handler, it's a bit messy and could break with API changes in the future
			 ... let's see if it works
	*/

	if request.APIGatewayProxyRequest.Path != "" {
		// This is an API Gateway Proxy request from Twilio
		var handler func(ctx context.Context, request events.APIGatewayProxyRequest, urlParams url.Values) (events.APIGatewayProxyResponse, error)
		var ok bool

		if handler, ok = h.apiMappings[request.APIGatewayProxyRequest.Path]; !ok {
			h.logger.Warn("Invalid request ", request.APIGatewayProxyRequest.Path)
			return Response{
				APIGatewayProxyResponse: events.APIGatewayProxyResponse{
					StatusCode: 400, Body: fmt.Sprintf("Invalid request path %v", request.APIGatewayProxyRequest.Path),
				},
			}, nil
		}

		// Validate the request and check signature
		values, err := url.ParseQuery(request.APIGatewayProxyRequest.Body)

		if err != nil {
			h.logger.Warn("Could not parse query ", err)
			return Response{
				APIGatewayProxyResponse: events.APIGatewayProxyResponse{
					StatusCode: 400, Body: "Could not parse body",
				},
			}, nil
		}

		if len(values) == 0 {
			return Response{
				APIGatewayProxyResponse: events.APIGatewayProxyResponse{
					StatusCode: 400, Body: "Empty request body",
				},
			}, nil
		}

		host := fmt.Sprintf("%v://%v", request.Headers["X-Forwarded-Proto"], request.Headers["Host"])
		url := fmt.Sprintf("/%v%v%v", request.RequestContext.Stage, request.Path, generateQueryString(request.APIGatewayProxyRequest.QueryStringParameters))

		err = h.twilio.ValidateRequest(request.Headers["X-Twilio-Signature"], host, h.TwilioAuthToken, url, values)
		if err != nil {
			h.logger.Warn(err)
			h.logger.Debug(request)

			return Response{
				APIGatewayProxyResponse: events.APIGatewayProxyResponse{
					StatusCode: 400, Body: "Invalid signature",
				},
			}, nil
		}

		// To make it easier to deal with internally we're just going to append the query string data to the post data
		for k, v := range request.APIGatewayProxyRequest.QueryStringParameters {
			values.Set(k, v)
		}

		resp, err := handler(ctx, request.APIGatewayProxyRequest, values)

		if err != nil {
			h.logger.Error("Error executing handler: ", err)
			return Response{APIGatewayProxyResponse: resp}, nil
		}

		return Response{
			APIGatewayProxyResponse: resp,
		}, nil
	} else if request.CloudWatchEvent.DetailType != "" {
		// This is a cloudwatch event trigger
		return Response{}, h.cloudwatchHandler(ctx, request.CloudWatchEvent)
	} else if request.SQSEvent.Records != nil {
		// This is an SQS handler
		errs := h.sqsHandler(ctx, request.SQSEvent)
		if len(errs) > 0 {
			for _, err := range errs {
				h.logger.Error("Error executing SQS handler: ", err)
			}
		}
		return Response{}, nil
	}

	return Response{}, errors.New("Invalid request")
}
