package handlers

import (
	aws "code.justin.tv/event-engineering/goldengate/pkg/aws/backend"
	awsBackend "code.justin.tv/event-engineering/goldengate/pkg/aws/backend/backendfakes"
	gg "code.justin.tv/event-engineering/goldengate/pkg/goldengate"
	jira "code.justin.tv/event-engineering/goldengate/pkg/jira/backend"
	jiraBackend "code.justin.tv/event-engineering/goldengate/pkg/jira/backend/backendfakes"
	loggingfakes "code.justin.tv/event-engineering/goldengate/pkg/logging/backend/backendfakes"
	pd "code.justin.tv/event-engineering/goldengate/pkg/pagerduty/backend"
	pdBackend "code.justin.tv/event-engineering/goldengate/pkg/pagerduty/backend/backendfakes"
	twilio "code.justin.tv/event-engineering/goldengate/pkg/twilio/backend"
	twilioBackend "code.justin.tv/event-engineering/goldengate/pkg/twilio/backend/backendfakes"
	"context"
	"github.com/aws/aws-lambda-go/events"
	goTwilio "github.com/kevinburke/twilio-go"
	"github.com/pkg/errors"
	"github.com/stretchr/testify/assert"
	"testing"
)

type mockBackend struct {
	pd           *pdBackend.FakeClient
	twilio       *twilioBackend.FakeClient
	jira         *jiraBackend.FakeClient
	aws          *awsBackend.FakeClient
	logger       *loggingfakes.FakeLogger
	handlersPriv *handlers
	handlersPub  Handlers
}

func (b *mockBackend) GetPagerdutyBackend() pd.Client {
	return b.pd
}

func (b *mockBackend) GetTwilioBackend() twilio.Client {
	return b.twilio
}

var jiraShouldError = false

func (b *mockBackend) GetJiraBackend() (jira.Client, error) {
	if jiraShouldError {
		return nil, errors.New("JIRA is BAD")
	}
	return b.jira, nil
}

var awsShouldError = false

func (b *mockBackend) GetAWSBackend() (aws.Client, error) {
	if awsShouldError {
		return nil, errors.New("AWS is BAD")
	}
	return b.aws, nil
}

func createMockBackend() (*mockBackend, error) {
	params := gg.Parameters{
		TwilioBridgeNumber:  "+1234567890",
		ConferenceName:      "TestConference",
		RecordingBucketName: "RecordingsLOL",
		HoldMusicURL:        "https://example.com/hold_music.xml",
		JiraHostname:        "jira.example.com",
		JiraConsumerKey:     "jira_consumer_key",
		EscalationPolicyID:  "ABC123",
		TeamID:              "DEF456",
		BlockedNumbers:      "+44123456098",
		BlockedPrefixes:     "+1800,",
		BatphoneNumbers:     "+44987654321",
		JiraEnabled:         true,
		PDEnabled:           true,
	}

	secureParams := gg.SecureParameters{
		TwilioAccountSid:              "ACCTSID",
		TwilioAuthToken:               "AUTHTOKEN",
		PagerDutyAPIKey:               "PDAPIKEY",
		PagerdutyTwilioIntegrationKey: "PDTWINTKEY",
		JiraToken:                     "JIRATOKEN",
		JiraTokenSecret:               "JIRATOKENSECRET",
		JiraPrivateKey:                "JIRAPRIVATEKEY",
	}

	backendResolver := &mockBackend{
		pd:     new(pdBackend.FakeClient),
		twilio: new(twilioBackend.FakeClient),
		jira:   new(jiraBackend.FakeClient),
		aws:    new(awsBackend.FakeClient),
		logger: new(loggingfakes.FakeLogger),
	}

	handlersPriv, err := newHandlers(backendResolver.logger, params, secureParams, backendResolver, 0)

	if err != nil {
		return nil, err
	}

	// We're not actually using this in tests but we want to call it to make sure it doesn't error
	_, err = New(backendResolver.logger, params, secureParams, backendResolver, 0)

	if err != nil {
		return nil, err
	}

	backendResolver.handlersPriv = handlersPriv
	backendResolver.handlersPub = handlersPriv // We're using the private handlers object here so that we can influence the settings at runtime during tests

	return backendResolver, nil
}

func TestErroringBackendCreation(t *testing.T) {
	a := assert.New(t)

	jiraShouldError = true
	backend, err := createMockBackend()
	a.Nil(backend)
	a.NotNil(err)
	a.EqualValues("JIRA is BAD", err.Error())

	jiraShouldError = false
	awsShouldError = true
	backend, err = createMockBackend()
	a.Nil(backend)
	a.NotNil(err)
	a.EqualValues("AWS is BAD", err.Error())

	// Make sure we don't break other tests
	awsShouldError = false
}

// Test all the main handler error code paths
// Success tests will be in the respective test files
func TestHandleErrors(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)
	a.NotNil(backend)

	ctx := context.TODO()

	request := Request{}

	// Test empty Request
	_, err = backend.handlersPub.Handle(ctx, request)
	a.NotNil(err)
	a.EqualValues("Invalid request", err.Error())

	// Test API Gateway request with invalid path
	request.APIGatewayProxyRequest = events.APIGatewayProxyRequest{
		Path: "/invalid",
	}

	resp, err := backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(400, resp.StatusCode)
	a.EqualValues("Invalid request path /invalid", resp.Body)

	// Test empty body
	request.APIGatewayProxyRequest.Path = "/new"
	resp, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(400, resp.StatusCode)
	a.EqualValues("Empty request body", resp.Body)

	// Test invalid body
	request.APIGatewayProxyRequest.Body = "abc=%5z"
	resp, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(400, resp.StatusCode)
	a.EqualValues("Could not parse body", resp.Body)

	// Test invalid signature
	request.APIGatewayProxyRequest.Body = "From=+1234567890"
	request.APIGatewayProxyRequest.Headers = map[string]string{
		"X-Forwarded-Proto":  "https",
		"Host":               "example.com",
		"X-Twilio-Signature": "blah",
	}

	// Use the real signature validation function to test with
	backend.twilio.GetExpectedSignatureStub = goTwilio.GetExpectedTwilioSignature

	resp, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(400, resp.StatusCode)
	a.EqualValues("Invalid signature", resp.Body)
}
