package handlers

import (
	"code.justin.tv/event-engineering/goldengate/pkg/pagerduty"
	pdBackend "code.justin.tv/event-engineering/goldengate/pkg/pagerduty/backend"
	"fmt"
	"github.com/aws/aws-lambda-go/events"
	"net/http"
	"net/url"
	"strings"
)

const createBridgeXML = `<?xml version="1.0" encoding="UTF-8"?><Response><Say>Please wait while we connect you. Calls will be recorded</Say><Dial><Conference statusCallback="%v/conferenceevent?InitialCallSid=%v" statusCallbackEvent="end,join" waitUrl="%v" waitMethod="GET" startConferenceOnEnter="%v" record="record-from-start" recordingStatusCallback="%v/recordingevent?InitialCallSid=%v">%v</Conference></Dial></Response>`

const newCallGatherXML = `<?xml version="1.0" encoding="UTF-8"?><Response><Gather numDigits="1"><Say>Welcome to the Twitch broadcast emergency phone bridge. Press 1 to be connected.</Say></Gather><Redirect /></Response>`

const joinConferenceXML = `<?xml version="1.0" encoding="UTF-8"?><Response><Say>Calls will be recorded</Say><Dial><Conference>%v</Conference></Dial></Response>`

const teamJoinBridgeXML = `<?xml version="1.0" encoding="UTF-8"?><Response><Dial><Conference startConferenceOnEnter="true">%v</Conference></Dial></Response>`

const teamGatherXML = `<?xml version="1.0" encoding="UTF-8"?><Response><Gather numDigits="1"><Say>Press 1 to be connected to the conference.</Say></Gather><Redirect /></Response>`

const blockedXML = `<?xml version="1.0" encoding="UTF-8"?><Response><Say>You have been blocked from calling this number. If this is in error, please contact your Twitch representative.</Say><Hangup/></Response>`

const noEventInProgressXML = `<?xml version="1.0" encoding="UTF-8"?><Response><Say>There is no conference in progress.</Say><Hangup/></Response>`

const errorXML = `<?xml version="1.0" encoding="UTF-8"?><Response><Say>There has been an error connecting you, please try again.</Say><Hangup/></Response>`

var blockedResponse = events.APIGatewayProxyResponse{
	StatusCode: http.StatusOK,
	Headers:    map[string]string{"Content-Type": "text/xml"},
	Body:       blockedXML,
}

var errorResponse = events.APIGatewayProxyResponse{
	StatusCode: http.StatusOK,
	Headers:    map[string]string{"Content-Type": "text/xml"},
	Body:       errorXML,
}

// Just so we don't have to type this over and over again.... we need an empty response to return in various situations
var er = events.APIGatewayProxyResponse{
	StatusCode: http.StatusOK,
	Headers:    map[string]string{"Content-Type": "text/plain"},
	Body:       "OK",
}

func isInSlice(a string, list []string) bool {
	for _, b := range list {
		if b == a {
			return true
		}
	}
	return false
}

func isInMap(a string, list map[string]string) bool {
	for _, b := range list {
		if b == a {
			return true
		}
	}
	return false
}

func generateRootURL(host, stage string) string {
	return fmt.Sprintf("https://%v/%v", host, stage)
}

func teamGatherResponse() events.APIGatewayProxyResponse {
	return events.APIGatewayProxyResponse{
		StatusCode: http.StatusOK,
		Headers:    map[string]string{"Content-Type": "text/xml"},
		Body:       teamGatherXML,
	}
}

func teamJoinConferenceResponse(conferenceName string) events.APIGatewayProxyResponse {
	return events.APIGatewayProxyResponse{
		StatusCode: http.StatusOK,
		Headers:    map[string]string{"Content-Type": "text/xml"},
		Body:       fmt.Sprintf(teamJoinBridgeXML, conferenceName),
	}
}

func newCallGatherResponse() events.APIGatewayProxyResponse {
	return events.APIGatewayProxyResponse{
		StatusCode: http.StatusOK,
		Headers:    map[string]string{"Content-Type": "text/xml"},
		Body:       fmt.Sprintf(newCallGatherXML),
	}
}

func joinConferenceResponse(conferenceName string) events.APIGatewayProxyResponse {
	return events.APIGatewayProxyResponse{
		StatusCode: http.StatusOK,
		Headers:    map[string]string{"Content-Type": "text/xml"},
		Body:       fmt.Sprintf(joinConferenceXML, conferenceName),
	}
}

func createConferenceResponse(rootURL, conferenceName, initialCallSid, holdMusicURL string, startOnEnter bool) events.APIGatewayProxyResponse {
	return events.APIGatewayProxyResponse{
		StatusCode: http.StatusOK,
		Headers:    map[string]string{"Content-Type": "text/xml"},
		Body:       fmt.Sprintf(createBridgeXML, rootURL, initialCallSid, holdMusicURL, startOnEnter, rootURL, initialCallSid, conferenceName),
	}
}

func createNoEventInProgressResponse() events.APIGatewayProxyResponse {
	return events.APIGatewayProxyResponse{
		StatusCode: http.StatusOK,
		Headers:    map[string]string{"Content-Type": "text/xml"},
		Body:       noEventInProgressXML,
	}
}

func getUserNumber(user *pdBackend.User) string {
	for _, cm := range user.ContactMethods {
		if cm.Type == pagerduty.PhoneNumberType {
			return fmt.Sprintf("+%v%v", cm.CountryCode, cm.Address)
		}
	}
	return ""
}

func generateQueryString(queryParameters map[string]string) string {
	qs := ""

	if len(queryParameters) > 0 {
		parts := make([]string, 0)
		for k, v := range queryParameters {
			part := ""
			if k != "" {
				part = url.QueryEscape(k)
			}

			if v != "" {
				part = part + "=" + url.QueryEscape(v)
			}

			if part != "" {
				parts = append(parts, part)
			}
		}

		if len(parts) > 0 {
			qs = "?" + strings.Join(parts, "&")
		}
	}

	return qs
}
