package handlers

import (
	"context"
	"fmt"
	"github.com/aws/aws-lambda-go/events"
	"net/url"
	"strings"
	"time"
)

func (h *handlers) isblocked(number string) bool {
	for _, blockedNum := range h.blockedNumbers {
		if strings.TrimSpace(blockedNum) == number {
			return true
		}
	}

	for _, blockedPrefix := range h.blockedPrefixes {
		// Just in case as this would end up blocking everything
		if blockedPrefix == "" {
			continue
		}

		if strings.HasPrefix(number, blockedPrefix) {
			return true
		}
	}

	return false
}

func (h *handlers) isBatphoneNumber(number string) bool {
	for _, batphoneNumber := range h.batphoneNumbers {
		if strings.TrimSpace(batphoneNumber) == number {
			return true
		}
	}

	return false
}

func (h *handlers) isTeamMember(number string) bool {
	isTeamMember := isInMap(number, h.getTeamUserNumbers(nil))
	isBatphoneNumber := h.isBatphoneNumber(number)

	return isTeamMember && !isBatphoneNumber
}

func (h *handlers) onNewCall(ctx context.Context, request events.APIGatewayProxyRequest, urlParams url.Values) (events.APIGatewayProxyResponse, error) {
	h.logger.Debug("NewCall fired", request.Body, request.QueryStringParameters)

	if h.isblocked(urlParams.Get("From")) {
		h.logger.Debug("Blocking call from ", urlParams.Get("From"))
		// This number is blocked
		return blockedResponse, nil
	}

	isTeamMember := h.isTeamMember(urlParams.Get("From"))
	isBatphoneNumber := h.isBatphoneNumber(urlParams.Get("From"))

	digits := urlParams.Get("Digits")

	if !isTeamMember && !isBatphoneNumber && (digits == "" || !validDigitsRegex.MatchString(digits)) {
		return newCallGatherResponse(), nil
	}

	// Check if there is already an open conference
	conferenceExists := h.twilio.IsConferenceOpen(ctx, h.ConferenceName)

	if conferenceExists {
		// If there is an open conference then we don't need to dial out to people or trigger a pagerduty alert, so just return the join conference response
		h.logger.Debug("Conference already exists, returning default join response")

		// If the person calling is a member of the team then return the team join conference response rather than the normal one
		// This covers the scenario of a team member dialling in to the conference manually (e.g. if they aren't on call but see the alert in slack)
		if isTeamMember {
			// The team join conference response will bring everyone off hold with the startConferenceOnEnter=true parameter
			return teamJoinConferenceResponse(h.ConferenceName), nil
		}

		// The join conference response will simply send the caller to the conference
		return joinConferenceResponse(h.ConferenceName), nil
	}

	// If the conference doesn't exist and the person calling is a member of the team then return a canned response
	if isTeamMember {
		return createNoEventInProgressResponse(), nil
	}

	if h.PDEnabled {
		// Trigger the pagerduty incident, including the caller number for quick reference
		triggerErr := h.pd.TriggerEvent(h.PagerdutyTwilioIntegrationKey, h.ConferenceName, fmt.Sprintf("Event Engineering Phone Bridge Triggered %v", urlParams.Get("From")))
		if triggerErr != nil {
			h.logger.Error("Error creating Pagerduty Incident", triggerErr)
		} else {
			h.logger.Debug("Created PD Incident")
		}
	} else {
		h.logger.Debug("PD is disabled, skipping")
	}

	rootURL := generateRootURL(request.Headers["Host"], request.RequestContext.Stage)

	// Dial out to the people who are on call at escalation level 1, we use the CallSid of the caller to uniquely identify this particular event
	// so it's passed around to various places to tie all the twilio callbacks together since otherwise they are generally stateless
	outgoingCalls, err := h.dialOnCalls(1, generatePickupURL(rootURL, urlParams.Get("CallSid")))
	if err != nil {
		return errorResponse, err
	}

	// Write the state to the DB, we use this state to tie everything together as various callbacks have varying amounts of this information available
	state := &State{
		Initiated:      time.Now(),
		TriggerCallSid: urlParams.Get("CallSid"),
		OutgoingCalls:  outgoingCalls,
		RootURL:        rootURL,
		EscalatedLevel: 1,
		LastEscalation: time.Now(),
	}

	err = h.putState(state)
	if err != nil {
		// If we've failed to put the state then there's little point in triggering the conference loop
		// most of the important functionality will still work like actually connecting the caller to a team member
		// however escalating after 60 seconds and tidying up incoming calls that didn't make it to the conference stage will not work
		h.logger.Error("Error writing state, no conference loop initiated ", err)
	} else {
		// Send the SQS loop message
		err := h.sendSQSLoop(&ConferenceLoopMessage{
			Initiated: time.Now(),
			CallSid:   state.TriggerCallSid,
		})

		if err != nil {
			h.logger.Error("Error sending SQS message ", err)
		}
	}

	// Return the full create-conference response to initialise the conference
	h.logger.Debug("Returning create-conference response")
	return createConferenceResponse(rootURL, h.ConferenceName, state.TriggerCallSid, h.HoldMusicURL, false), nil
}
