package handlers

import (
	goPd "code.justin.tv/event-engineering/goldengate/pkg/pagerduty/backend"
	"context"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/sqs"
	goTwilio "github.com/kevinburke/twilio-go"
	pdEvents "github.com/marcw/pagerduty"
	"github.com/pkg/errors"
	"github.com/stretchr/testify/assert"
	"net/url"
	"testing"
	"time"
)

type conferenceIterator struct{}

func (i *conferenceIterator) Next(context.Context) (*goTwilio.ConferencePage, error) {
	return &goTwilio.ConferencePage{
		Conferences: []*goTwilio.Conference{
			&goTwilio.Conference{
				FriendlyName: "hello",
			},
		},
	}, nil
}

type emptyConferenceIterator struct{}

func (i *emptyConferenceIterator) Next(context.Context) (*goTwilio.ConferencePage, error) {
	return nil, goTwilio.NoMoreResults
}

var newCallRequestWithDigits = Request{
	APIGatewayProxyRequest: events.APIGatewayProxyRequest{
		Headers: map[string]string{"Host": "example.com"},
		RequestContext: events.APIGatewayProxyRequestContext{
			Stage: "test",
		},
		Path: "/new",
		Body: "From=%2B44098765432&CallSid=CALL_SID&Digits=1",
	},
}

var conferenceDoesntExist = func(start time.Time, end time.Time, data url.Values) goTwilio.ConferencePageIterator {
	return &emptyConferenceIterator{}
}

var onCallList = func(escalationPolicyID string) (*goPd.ListOnCallsResponse, error) {
	return &goPd.ListOnCallsResponse{
		OnCalls: []*goPd.OnCall{
			&goPd.OnCall{
				EscalationLevel: 1,
				User: &goPd.OnCallUser{
					ID: "USER1",
					ContactMethods: []*goPd.ContactMethodReference{
						&goPd.ContactMethodReference{
							ID:   "CM1",
							Type: "phone_contact_method_reference",
						},
					},
				},
			},
			&goPd.OnCall{
				EscalationLevel: 1,
				User: &goPd.OnCallUser{
					ID: "USER2",
					ContactMethods: []*goPd.ContactMethodReference{
						&goPd.ContactMethodReference{
							ID:   "CM2",
							Type: "phone_contact_method_reference",
						},
					},
				},
			},
			&goPd.OnCall{
				EscalationLevel: 2,
				User: &goPd.OnCallUser{
					ID: "USER3",
					ContactMethods: []*goPd.ContactMethodReference{
						&goPd.ContactMethodReference{
							ID:   "CM3",
							Type: "phone_contact_method_reference",
						},
					},
				},
			},
		},
	}, nil
}

var teamUsers = func(teamID string) (*goPd.ListUsersResponse, error) {
	return &goPd.ListUsersResponse{
		Users: []*goPd.User{
			&goPd.User{
				ID: "USER1",
				ContactMethods: []*goPd.ContactMethod{
					&goPd.ContactMethod{
						ID:          "CM1",
						Type:        "phone_contact_method",
						Address:     "1234567890",
						CountryCode: 44,
					},
				},
			},
			&goPd.User{
				ID: "USER2",
			},
			&goPd.User{
				ID: "USER3",
				ContactMethods: []*goPd.ContactMethod{
					&goPd.ContactMethod{
						ID:          "CM3",
						Type:        "phone_contact_method",
						Address:     "98979695949",
						CountryCode: 44,
					},
				},
			},
		},
	}, nil
}

var createCall = func(from string, to string, u *url.URL) (*goTwilio.Call, error) {
	return &goTwilio.Call{
		Sid: "ABC123",
	}, nil
}

func TestBlockedNumberResponse(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	// We'll test PD later
	backend.handlersPriv.PDEnabled = false

	ctx := context.TODO()

	// Test blocked number
	request := Request{
		APIGatewayProxyRequest: events.APIGatewayProxyRequest{
			Headers: map[string]string{"Host": "example.com"},
			RequestContext: events.APIGatewayProxyRequestContext{
				Stage: "test",
			},
			Path: "/new",
			Body: "From=%2B44123456098&CallSid=CALL_SID",
		},
	}

	resp, err := backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues("text/xml", resp.Headers["Content-Type"])
	a.EqualValues(`<?xml version="1.0" encoding="UTF-8"?><Response><Say>You have been blocked from calling this number. If this is in error, please contact your Twitch representative.</Say><Hangup/></Response>`, resp.Body)
}

func TestBlockedPrefixResponse(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	// We'll test PD later
	backend.handlersPriv.PDEnabled = false

	ctx := context.TODO()

	// Test blocked number
	request := Request{
		APIGatewayProxyRequest: events.APIGatewayProxyRequest{
			Headers: map[string]string{"Host": "example.com"},
			RequestContext: events.APIGatewayProxyRequestContext{
				Stage: "test",
			},
			Path: "/new",
			Body: "From=%2B18003456098&CallSid=CALL_SID",
		},
	}

	resp, err := backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues("text/xml", resp.Headers["Content-Type"])
	a.EqualValues(`<?xml version="1.0" encoding="UTF-8"?><Response><Say>You have been blocked from calling this number. If this is in error, please contact your Twitch representative.</Say><Hangup/></Response>`, resp.Body)
}

func TestNewCallFromTeam(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	// We'll test PD later
	backend.handlersPriv.PDEnabled = false

	ctx := context.TODO()

	backend.pd.ListTeamUsersStub = teamUsers
	backend.twilio.GetConferencesStub = func(start time.Time, end time.Time, data url.Values) goTwilio.ConferencePageIterator {
		return &emptyConferenceIterator{}
	}

	// Test team creating conference
	request := Request{
		APIGatewayProxyRequest: events.APIGatewayProxyRequest{
			Headers: map[string]string{"Host": "example.com"},
			RequestContext: events.APIGatewayProxyRequestContext{
				Stage: "test",
			},
			Path: "/new",
			Body: "From=%2B441234567890&CallSid=CALL_SID",
		},
	}

	resp, err := backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues("text/xml", resp.Headers["Content-Type"])
	a.EqualValues(`<?xml version="1.0" encoding="UTF-8"?><Response><Say>There is no conference in progress.</Say><Hangup/></Response>`, resp.Body)
}

func TestConferenceExists(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	// We'll test PD later
	backend.handlersPriv.PDEnabled = false

	ctx := context.TODO()

	// Test conference already exists
	request := newCallRequestWithDigits

	backend.twilio.GetConferencesStub = func(start time.Time, end time.Time, data url.Values) goTwilio.ConferencePageIterator {
		return &conferenceIterator{}
	}

	resp, err := backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues("text/xml", resp.Headers["Content-Type"])
	a.EqualValues(`<?xml version="1.0" encoding="UTF-8"?><Response><Say>Calls will be recorded</Say><Dial><Conference>TestConference</Conference></Dial></Response>`, resp.Body)
	a.EqualValues(0, backend.logger.ErrorCallCount())
}

func TestTeamDialledInManually(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	// We'll test PD later
	backend.handlersPriv.PDEnabled = false

	ctx := context.TODO()

	// Test team member dialled in manually to conference
	request := Request{
		APIGatewayProxyRequest: events.APIGatewayProxyRequest{
			Headers: map[string]string{"Host": "example.com"},
			RequestContext: events.APIGatewayProxyRequestContext{
				Stage: "test",
			},
			Path: "/new",
			Body: "From=%2B441234567890&CallSid=CALL_SID",
		},
	}

	backend.pd.ListTeamUsersStub = teamUsers

	backend.twilio.GetConferencesStub = func(start time.Time, end time.Time, data url.Values) goTwilio.ConferencePageIterator {
		return &conferenceIterator{}
	}

	resp, err := backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues("text/xml", resp.Headers["Content-Type"])
	a.EqualValues(`<?xml version="1.0" encoding="UTF-8"?><Response><Dial><Conference startConferenceOnEnter="true">TestConference</Conference></Dial></Response>`, resp.Body)
	a.EqualValues(0, backend.logger.ErrorCallCount())
}

func TestErrorGettingOncalls(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	// We'll test PD later
	backend.handlersPriv.PDEnabled = false

	ctx := context.TODO()

	// Test error retrieving oncall numbers
	request := newCallRequestWithDigits
	backend.twilio.GetConferencesStub = conferenceDoesntExist

	backend.pd.ListOnCallsStub = func(escalationPolicyID string) (*goPd.ListOnCallsResponse, error) {
		return nil, errors.New("PD list error")
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Error executing handler: ", args[0])
		a.EqualValues("Error getting oncall telephone numbers: Failed to get Oncall Users: PD list error", error(args[1].(error)).Error())
	}

	resp, err := backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues("text/xml", resp.Headers["Content-Type"])
	a.EqualValues(`<?xml version="1.0" encoding="UTF-8"?><Response><Say>There has been an error connecting you, please try again.</Say><Hangup/></Response>`, resp.Body)
	a.EqualValues(1, backend.logger.ErrorCallCount())
}

func TestEmptyTeamUsersList(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	// We'll test PD later
	backend.handlersPriv.PDEnabled = false

	ctx := context.TODO()

	// Test empty team users list
	request := newCallRequestWithDigits
	backend.twilio.GetConferencesStub = conferenceDoesntExist
	backend.pd.ListOnCallsStub = onCallList

	backend.pd.ListTeamUsersStub = func(teamID string) (*goPd.ListUsersResponse, error) {
		return &goPd.ListUsersResponse{
			Users: []*goPd.User{},
		}, nil
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Error executing handler: ", args[0])
		a.EqualValues("Zero oncall numbers could be found at escalation level 1", error(args[1].(error)).Error())
	}

	resp, err := backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues("text/xml", resp.Headers["Content-Type"])
	a.EqualValues(`<?xml version="1.0" encoding="UTF-8"?><Response><Say>There has been an error connecting you, please try again.</Say><Hangup/></Response>`, resp.Body)
	a.EqualValues(1, backend.logger.ErrorCallCount())
}

func TestZeroSuccessfulDials(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	// We'll test PD later
	backend.handlersPriv.PDEnabled = false

	ctx := context.TODO()

	// Test zero successful dials
	request := newCallRequestWithDigits
	backend.twilio.GetConferencesStub = conferenceDoesntExist
	backend.pd.ListOnCallsStub = onCallList
	backend.pd.ListTeamUsersStub = teamUsers

	backend.twilio.CallStub = func(from string, to string, u *url.URL) (*goTwilio.Call, error) {
		return nil, errors.New("Error making call")
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Error executing handler: ", args[0])
		a.EqualValues("Zero successful dials", error(args[1].(error)).Error())
	}

	resp, err := backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues("text/xml", resp.Headers["Content-Type"])
	a.EqualValues(`<?xml version="1.0" encoding="UTF-8"?><Response><Say>There has been an error connecting you, please try again.</Say><Hangup/></Response>`, resp.Body)
	a.EqualValues(1, backend.logger.ErrorCallCount())
}

func TestErrorSendingSQSMessage(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	// We'll test PD later
	backend.handlersPriv.PDEnabled = false

	ctx := context.TODO()

	// Test error sending SQS message
	request := newCallRequestWithDigits
	backend.twilio.GetConferencesStub = conferenceDoesntExist
	backend.pd.ListOnCallsStub = onCallList
	backend.pd.ListTeamUsersStub = teamUsers
	backend.twilio.CallStub = createCall
	backend.aws.SendSQSMessageStub = func(input *sqs.SendMessageInput) (*sqs.SendMessageOutput, error) {
		return nil, errors.New("SQS will not allow it")
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Error sending SQS message ", args[0])
		a.EqualValues("Error sending message for SQS send: SQS will not allow it", error(args[1].(error)).Error())
	}

	resp, err := backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues("text/xml", resp.Headers["Content-Type"])
	a.EqualValues(`<?xml version="1.0" encoding="UTF-8"?><Response><Say>Please wait while we connect you. Calls will be recorded</Say><Dial><Conference statusCallback="https://example.com/test/conferenceevent?InitialCallSid=CALL_SID" statusCallbackEvent="end,join" waitUrl="https://example.com/hold_music.xml" waitMethod="GET" startConferenceOnEnter="false" record="record-from-start" recordingStatusCallback="https://example.com/test/recordingevent?InitialCallSid=CALL_SID">TestConference</Conference></Dial></Response>`, resp.Body)
	a.EqualValues(1, backend.logger.ErrorCallCount())
	a.EqualValues(1, backend.aws.SendSQSMessageCallCount())
}

func TestSuccessfulDial(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	// We'll test PD later
	backend.handlersPriv.PDEnabled = false

	ctx := context.TODO()

	// Test successful dial
	request := newCallRequestWithDigits
	backend.twilio.GetConferencesStub = conferenceDoesntExist
	backend.pd.ListOnCallsStub = onCallList
	backend.pd.ListTeamUsersStub = teamUsers
	backend.twilio.CallStub = createCall

	resp, err := backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues("text/xml", resp.Headers["Content-Type"])
	a.EqualValues(`<?xml version="1.0" encoding="UTF-8"?><Response><Say>Please wait while we connect you. Calls will be recorded</Say><Dial><Conference statusCallback="https://example.com/test/conferenceevent?InitialCallSid=CALL_SID" statusCallbackEvent="end,join" waitUrl="https://example.com/hold_music.xml" waitMethod="GET" startConferenceOnEnter="false" record="record-from-start" recordingStatusCallback="https://example.com/test/recordingevent?InitialCallSid=CALL_SID">TestConference</Conference></Dial></Response>`, resp.Body)
	a.EqualValues(0, backend.logger.ErrorCallCount())
	a.EqualValues(1, backend.aws.SendSQSMessageCallCount())
}

func TestNewCallFromTeamBatphone(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	// We'll test PD later
	backend.handlersPriv.PDEnabled = false

	ctx := context.TODO()

	backend.pd.ListTeamUsersStub = teamUsers
	backend.twilio.GetConferencesStub = func(start time.Time, end time.Time, data url.Values) goTwilio.ConferencePageIterator {
		return &emptyConferenceIterator{}
	}

	// Test batphone creating conference
	request := Request{
		APIGatewayProxyRequest: events.APIGatewayProxyRequest{
			Headers: map[string]string{"Host": "example.com"},
			RequestContext: events.APIGatewayProxyRequestContext{
				Stage: "test",
			},
			Path: "/new",
			Body: "From=%2B44987654321&CallSid=CALL_SID",
		},
	}
	backend.twilio.GetConferencesStub = conferenceDoesntExist
	backend.pd.ListOnCallsStub = onCallList
	backend.pd.ListTeamUsersStub = teamUsers
	backend.twilio.CallStub = createCall

	resp, err := backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues("text/xml", resp.Headers["Content-Type"])
	a.EqualValues(`<?xml version="1.0" encoding="UTF-8"?><Response><Say>Please wait while we connect you. Calls will be recorded</Say><Dial><Conference statusCallback="https://example.com/test/conferenceevent?InitialCallSid=CALL_SID" statusCallbackEvent="end,join" waitUrl="https://example.com/hold_music.xml" waitMethod="GET" startConferenceOnEnter="false" record="record-from-start" recordingStatusCallback="https://example.com/test/recordingevent?InitialCallSid=CALL_SID">TestConference</Conference></Dial></Response>`, resp.Body)
	a.EqualValues(0, backend.logger.ErrorCallCount())
	a.EqualValues(1, backend.aws.SendSQSMessageCallCount())
}

func TestNewCallWithoutDigits(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	// We'll test PD later
	backend.handlersPriv.PDEnabled = false

	ctx := context.TODO()

	backend.pd.ListTeamUsersStub = teamUsers
	backend.twilio.GetConferencesStub = func(start time.Time, end time.Time, data url.Values) goTwilio.ConferencePageIterator {
		return &emptyConferenceIterator{}
	}

	// Test initial dial from user without pressing digits
	request := Request{
		APIGatewayProxyRequest: events.APIGatewayProxyRequest{
			Headers: map[string]string{"Host": "example.com"},
			RequestContext: events.APIGatewayProxyRequestContext{
				Stage: "test",
			},
			Path: "/new",
			Body: "From=%2B44098765432&CallSid=CALL_SID",
		},
	}
	backend.twilio.GetConferencesStub = conferenceDoesntExist
	backend.pd.ListOnCallsStub = onCallList
	backend.pd.ListTeamUsersStub = teamUsers
	backend.twilio.CallStub = createCall

	resp, err := backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues("text/xml", resp.Headers["Content-Type"])
	a.EqualValues(`<?xml version="1.0" encoding="UTF-8"?><Response><Gather numDigits="1"><Say>Welcome to the Twitch broadcast emergency phone bridge. Press 1 to be connected.</Say></Gather><Redirect /></Response>`, resp.Body)
	a.EqualValues(0, backend.logger.ErrorCallCount())
	a.EqualValues(0, backend.aws.SendSQSMessageCallCount())
}

func TestErrorPuttingState(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	// We'll test PD later
	backend.handlersPriv.PDEnabled = false

	ctx := context.TODO()

	// Test successful dial
	request := newCallRequestWithDigits
	backend.twilio.GetConferencesStub = conferenceDoesntExist
	backend.pd.ListOnCallsStub = onCallList
	backend.pd.ListTeamUsersStub = teamUsers
	backend.twilio.CallStub = createCall

	backend.aws.DDBPutItemStub = func(input *dynamodb.PutItemInput) (*dynamodb.PutItemOutput, error) {
		return nil, errors.New("DDB broke man")
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Error writing state, no conference loop initiated ", args[0])
		a.EqualValues("DDB broke man", error(args[1].(error)).Error())
	}

	resp, err := backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues("text/xml", resp.Headers["Content-Type"])
	a.EqualValues(`<?xml version="1.0" encoding="UTF-8"?><Response><Say>Please wait while we connect you. Calls will be recorded</Say><Dial><Conference statusCallback="https://example.com/test/conferenceevent?InitialCallSid=CALL_SID" statusCallbackEvent="end,join" waitUrl="https://example.com/hold_music.xml" waitMethod="GET" startConferenceOnEnter="false" record="record-from-start" recordingStatusCallback="https://example.com/test/recordingevent?InitialCallSid=CALL_SID">TestConference</Conference></Dial></Response>`, resp.Body)
	a.EqualValues(1, backend.logger.ErrorCallCount())
	a.EqualValues(0, backend.aws.SendSQSMessageCallCount())
}

func TestPDTriggerError(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test erroring Pagerduty event trigger
	request := newCallRequestWithDigits
	backend.twilio.GetConferencesStub = conferenceDoesntExist
	backend.pd.ListOnCallsStub = onCallList
	backend.pd.ListTeamUsersStub = teamUsers
	backend.twilio.CallStub = createCall

	backend.pd.SubmitEventStub = func(event *pdEvents.Event) (response *pdEvents.Response, statusCode int, err error) {
		return nil, 0, errors.New("Error creating event")
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Error creating Pagerduty Incident", args[0])
		a.EqualValues("Failed to submit PD event: Error creating event", error(args[1].(error)).Error())
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.pd.SubmitEventCallCount())
	a.EqualValues(1, backend.logger.ErrorCallCount())
	a.EqualValues(1, backend.aws.SendSQSMessageCallCount())
}

func TestPDTriggerSuccess(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test successful PD trigger
	request := newCallRequestWithDigits
	backend.twilio.GetConferencesStub = conferenceDoesntExist
	backend.pd.ListOnCallsStub = onCallList
	backend.pd.ListTeamUsersStub = teamUsers
	backend.twilio.CallStub = createCall

	backend.pd.SubmitEventStub = func(event *pdEvents.Event) (response *pdEvents.Response, statusCode int, err error) {
		return nil, 200, nil
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.pd.SubmitEventCallCount())
	a.EqualValues(0, backend.logger.ErrorCallCount())
	a.EqualValues(1, backend.aws.SendSQSMessageCallCount())
}
