package handlers

import (
	"context"
	"fmt"
	"github.com/aws/aws-lambda-go/events"
	"net/url"
	"regexp"
)

// We say "press 1 to connect", but in reality we'll accept any number, we just want to make sure it's human rather than voicemail
var validDigitsRegex = regexp.MustCompile("^[0-9]+$")

func generatePickupURL(rootURL, initialCallSid string) string {
	return fmt.Sprintf("%v/pickup?InitialCallSid=%v", rootURL, initialCallSid)
}

func (h *handlers) onPickup(ctx context.Context, request events.APIGatewayProxyRequest, urlParams url.Values) (events.APIGatewayProxyResponse, error) {
	h.logger.Debug("Pickup fired", request.Body)

	/*
		When the oncall engineer picks up we don't know if it's them picking up, or if it's their voicemail, so we're putting in a step where they have to press 1 on the keypad to connect to the call
		We're using the same handler for both events, the only difference being that there will be a "Digits" query string entry if we've come via the Gather response and the oncall
		engineer has pressed 1
	*/

	digits := urlParams.Get("Digits")

	if digits != "" && validDigitsRegex.MatchString(digits) {
		/*
			It's possible for the person oncall to pick up before the conference has been created by the caller as there is a short message before joining the conference
			In this scenario we want to return the full create-conference response with all the registered callbacks
			We have the initial call Sid from the request so we can use that to grab the state and then generate the right response
		*/
		conferenceExists := h.twilio.IsConferenceOpen(ctx, h.ConferenceName)
		if conferenceExists {
			return teamJoinConferenceResponse(h.ConferenceName), nil
		}

		rootURL := generateRootURL(request.Headers["Host"], request.RequestContext.Stage)
		return createConferenceResponse(rootURL, h.ConferenceName, urlParams.Get("InitialCallSid"), h.HoldMusicURL, true), nil
	}

	// This response will prompt the user to press 1 on the keypad making sure it's not a voicemail
	return teamGatherResponse(), nil
}
