package handlers

import (
	"context"
	"github.com/aws/aws-lambda-go/events"
	goTwilio "github.com/kevinburke/twilio-go"
	"github.com/stretchr/testify/assert"
	"net/url"
	"testing"
	"time"
)

func TestGeneratePickupURL(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	rootURL := "https://google.com"
	initialCallSid := "initial_call_sid"

	pickupURL := generatePickupURL(rootURL, initialCallSid)
	a.EqualValues("https://google.com/pickup?InitialCallSid=initial_call_sid", pickupURL)
}

func TestPickupResponse(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	backend.twilio.GetConferencesStub = func(start time.Time, end time.Time, data url.Values) goTwilio.ConferencePageIterator {
		return &conferenceIterator{}
	}

	// Test pickup
	request := Request{
		APIGatewayProxyRequest: events.APIGatewayProxyRequest{
			Headers: map[string]string{"Host": "example.com"},
			RequestContext: events.APIGatewayProxyRequestContext{
				Stage: "test",
			},
			Path: "/pickup",
			Body: "From=+44123456098",
		},
	}

	resp, err := backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues("text/xml", resp.Headers["Content-Type"])
	a.EqualValues(`<?xml version="1.0" encoding="UTF-8"?><Response><Gather numDigits="1"><Say>Press 1 to be connected to the conference.</Say></Gather><Redirect /></Response>`, resp.Body)
}

func TestGatherResponse(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	backend.twilio.GetConferencesStub = func(start time.Time, end time.Time, data url.Values) goTwilio.ConferencePageIterator {
		return &conferenceIterator{}
	}

	// Test pickup
	request := Request{
		APIGatewayProxyRequest: events.APIGatewayProxyRequest{
			Headers: map[string]string{"Host": "example.com"},
			RequestContext: events.APIGatewayProxyRequestContext{
				Stage: "test",
			},
			Path: "/pickup",
			Body: "From=+44123456098&Digits=1",
		},
	}

	resp, err := backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues("text/xml", resp.Headers["Content-Type"])
	a.EqualValues(`<?xml version="1.0" encoding="UTF-8"?><Response><Dial><Conference startConferenceOnEnter="true">TestConference</Conference></Dial></Response>`, resp.Body)
}

func TestPickedUpSuperFastResponse(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	backend.twilio.GetConferencesStub = func(start time.Time, end time.Time, data url.Values) goTwilio.ConferencePageIterator {
		return &emptyConferenceIterator{}
	}

	// Test pickup
	request := Request{
		APIGatewayProxyRequest: events.APIGatewayProxyRequest{
			Headers: map[string]string{"Host": "example.com"},
			RequestContext: events.APIGatewayProxyRequestContext{
				Stage: "test",
			},
			Path: "/pickup",
			QueryStringParameters: map[string]string{
				"InitialCallSid": "initial_call_sid",
			},
			Body: "From=+44123456098&Digits=1",
		},
	}

	resp, err := backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues("text/xml", resp.Headers["Content-Type"])
	a.EqualValues(`<?xml version="1.0" encoding="UTF-8"?><Response><Say>Please wait while we connect you. Calls will be recorded</Say><Dial><Conference statusCallback="https://example.com/test/conferenceevent?InitialCallSid=initial_call_sid" statusCallbackEvent="end,join" waitUrl="https://example.com/hold_music.xml" waitMethod="GET" startConferenceOnEnter="true" record="record-from-start" recordingStatusCallback="https://example.com/test/recordingevent?InitialCallSid=initial_call_sid">TestConference</Conference></Dial></Response>`, resp.Body)
}
