package handlers

import (
	"code.justin.tv/event-engineering/goldengate/pkg/jira"
	"context"
	"fmt"
	"github.com/aws/aws-lambda-go/events"
	"net/url"
	"regexp"
)

var recordingRegex = regexp.MustCompile("^.+/([^/]+)$")

func (h *handlers) onRecordingEvent(ctx context.Context, request events.APIGatewayProxyRequest, urlParams url.Values) (events.APIGatewayProxyResponse, error) {
	h.logger.Debug("RecordingEvent fired", request.Body)

	// Values can be "in-progress", "completed" or "failed"
	if urlParams.Get("RecordingStatus") == "in-progress" {
		h.logger.Warn("Recording in progress, skipping")
		return er, nil
	}

	presignedURL := ""

	if urlParams.Get("RecordingStatus") == "completed" {
		var fullRecordingName = urlParams.Get("RecordingUrl") + ".mp3"

		// Grab the recording from twilio
		recording, err := h.twilio.RetrieveRecording(fullRecordingName)
		if err != nil {
			return er, err
		}

		// Format the S3 key
		s3Key := fmt.Sprintf("%v/%v", urlParams.Get("ConferenceSid"), recordingRegex.ReplaceAllString(urlParams.Get("RecordingUrl"), "$1"))

		// Upload the recording and return a presigned URL valid for 7 days
		presignedURL, err = h.aws.UploadRecording(recording, h.RecordingBucketName, s3Key)

		if err != nil {
			h.logger.Error("Error uploading recording", err)
		}
	} else {
		// In the failed state we still want to do the rest of the work, so just log and put something useful in the URL field
		h.logger.Warnf("Recording is not complete, status %v", urlParams.Get("RecordingStatus"))
		presignedURL = fmt.Sprintf("Recording status: %v", urlParams.Get("RecordingStatus"))
	}

	initialCaller := "unknown"

	// Grab the state so we can add the initial caller number to the JIRA ticket
	state, err := h.getStateFromConference(urlParams.Get("ConferenceSid"))
	if err != nil {
		h.logger.Errorf("Error getting state for conference sid %v %v", urlParams.Get("ConferenceSid"), err)
	} else {
		// Grab the call that triggered the event
		initialCall, err := h.twilio.GetCall(ctx, state.TriggerCallSid)
		if err != nil {
			h.logger.Error("Error getting initial call information ", err)
		} else {
			initialCaller = string(initialCall.From)
		}

		// We don't need the state in the DB any more so delete it
		err = h.deleteState(state)
		if err != nil {
			h.logger.Errorf("Error deleting state for call sid %v %v", state.TriggerCallSid, err)
		}
	}

	if h.JiraEnabled {
		// Create the JIRA ticket with all the info
		ticket := &jira.Ticket{
			ProjectKey:    "BROAD",
			IssueType:     "Event Incident",
			Summary:       fmt.Sprintf("EE phone bridge incident (%v)", urlParams.Get("ConferenceSid")),
			Description:   "Event Engineering phone bridge incident (recording link expires in 7 days, if you need to access the recording after this time, email eventengineering@twitch.tv)",
			Caller:        initialCaller,
			ConferenceSid: urlParams.Get("ConferenceSid"),
			RecordingSid:  urlParams.Get("RecordingSid"),
			RecordingURL:  presignedURL,
		}
		err = h.jira.CreateTicket(ticket)

		if err != nil {
			return er, err
		}

		h.logger.Debugf("JIRA issue created with Key %v", ticket.TicketKey)
	} else {
		h.logger.Debug("Jira is disabled, skipping")
	}

	return er, nil
}
