package handlers

import (
	"context"
	"fmt"
	goJira "github.com/andygrunwald/go-jira"
	"github.com/aws/aws-lambda-go/events"
	awsRequest "github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3manager"
	goTwilio "github.com/kevinburke/twilio-go"
	"github.com/pkg/errors"
	"github.com/stretchr/testify/assert"
	"io"
	"net/http"
	"net/url"
	"testing"
)

var s3GetObjectRequestStub = func(input *s3.GetObjectInput) (req *awsRequest.Request, output *s3.GetObjectOutput) {
	return &awsRequest.Request{
		Operation: &awsRequest.Operation{
			Name:       "GetObject",
			HTTPMethod: "GET",
			HTTPPath:   fmt.Sprintf("/{%v/%v", input.Bucket, input.Key),
		},
		HTTPRequest: &http.Request{
			URL: &url.URL{
				Scheme: "https",
				Host:   "google.com",
				Path:   "test",
			},
		},
	}, nil
}

var getCallSuccess = func(ctx context.Context, sid string) (*goTwilio.Call, error) {
	return &goTwilio.Call{
		Sid:  "ABC123",
		From: "+441230984657",
		To:   "+1975638394",
	}, nil
}

func TestUnknownRecordingStatus(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	backend.aws.DDBQueryStub = func(input *dynamodb.QueryInput) (*dynamodb.QueryOutput, error) {
		return &dynamodb.QueryOutput{
			Items: []map[string]*dynamodb.AttributeValue{},
		}, nil
	}

	// Test unknown recording status
	request := Request{
		APIGatewayProxyRequest: events.APIGatewayProxyRequest{
			Path: "/recordingevent",
			Body: "RecordingStatus=in-progress",
		},
	}

	backend.logger.WarnStub = func(args ...interface{}) {
		a.EqualValues("Recording in progress, skipping", args[0])
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.WarnCallCount())
}

func TestFailedRecording(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test failed recording
	request := Request{
		APIGatewayProxyRequest: events.APIGatewayProxyRequest{
			Path: "/recordingevent",
			Body: "RecordingStatus=failed",
		},
	}

	backend.aws.S3GetObjectRequestStub = s3GetObjectRequestStub
	backend.aws.DDBQueryStub = getStateByConfSuccess
	backend.twilio.GetCallStub = getCallSuccess

	backend.jira.IssueCreateStub = func(issue *goJira.Issue) (*goJira.Issue, *goJira.Response, error) {
		return &goJira.Issue{
			ID: "1234567",
		}, nil, nil
	}

	backend.jira.IssueGetStub = func(issueID string, options *goJira.GetQueryOptions) (*goJira.Issue, *goJira.Response, error) {
		return &goJira.Issue{
			Key: "TICKET-1337",
		}, nil, nil
	}

	backend.logger.DebugfStub = func(format string, args ...interface{}) {
		a.EqualValues("JIRA issue created with Key %v", format)
		a.EqualValues("TICKET-1337", string(args[0].(string)))
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.DebugfCallCount())
	a.EqualValues(0, backend.logger.ErrorCallCount())
}

func TestFailedToRetrieveRecording(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test failure to retrieve recording
	request := Request{
		APIGatewayProxyRequest: events.APIGatewayProxyRequest{
			Path: "/recordingevent",
			Body: "RecordingStatus=completed&ConferenceSid=conf_sid",
		},
	}

	backend.twilio.RetrieveRecordingStub = func(url string) (io.ReadCloser, error) {
		return nil, errors.New("error downloading file")
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Error executing handler: ", args[0])
		a.EqualValues("error downloading file", error(args[1].(error)).Error())
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorCallCount())
}

func TestAWSUploadFailure(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test failure to upload recording to AWS
	request := Request{
		APIGatewayProxyRequest: events.APIGatewayProxyRequest{
			Path: "/recordingevent",
			Body: "RecordingStatus=completed&ConferenceSid=conf_sid",
		},
	}

	backend.aws.DDBQueryStub = getStateByConfSuccess
	backend.twilio.GetCallStub = getCallSuccess

	backend.aws.S3UploadStub = func(input *s3manager.UploadInput, options ...func(*s3manager.Uploader)) (*s3manager.UploadOutput, error) {
		return nil, errors.New("S3 upload error")
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Error uploading recording", args[0])
		a.EqualValues("Failed to upload recording: S3 upload error", error(args[1].(error)).Error())
	}

	backend.handlersPriv.JiraEnabled = false

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorCallCount())
}

func TestFailureToGetState(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test failure to get state from conference Sid
	request := Request{
		APIGatewayProxyRequest: events.APIGatewayProxyRequest{
			Path: "/recordingevent",
			Body: "RecordingStatus=completed&ConferenceSid=conf_sid",
		},
	}

	backend.aws.S3GetObjectRequestStub = s3GetObjectRequestStub

	backend.aws.DDBQueryStub = func(input *dynamodb.QueryInput) (*dynamodb.QueryOutput, error) {
		return nil, errors.New("DDB broke :(")
	}

	backend.logger.ErrorfStub = func(format string, args ...interface{}) {
		a.EqualValues("Error getting state for conference sid %v %v", format)
		a.EqualValues("conf_sid", args[0])
		a.EqualValues("DDB broke :(", error(args[1].(error)).Error())
	}

	backend.handlersPriv.JiraEnabled = false

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorfCallCount())
}

func TestFailureToGetInitialCallInfo(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test failure to get state from conference Sid
	request := Request{
		APIGatewayProxyRequest: events.APIGatewayProxyRequest{
			Path: "/recordingevent",
			Body: "RecordingStatus=completed&ConferenceSid=conf_sid",
		},
	}

	backend.aws.S3GetObjectRequestStub = s3GetObjectRequestStub
	backend.aws.DDBQueryStub = getStateByConfSuccess

	backend.twilio.GetCallStub = func(ctx context.Context, sid string) (*goTwilio.Call, error) {
		return nil, errors.New("ripperoni")
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Error getting initial call information ", args[0])
		a.EqualValues("ripperoni", error(args[1].(error)).Error())
	}

	backend.handlersPriv.JiraEnabled = false

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorCallCount())
}

func TestFailureToDeleteState(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test failure to get state from conference Sid
	request := Request{
		APIGatewayProxyRequest: events.APIGatewayProxyRequest{
			Path: "/recordingevent",
			Body: "RecordingStatus=completed&ConferenceSid=conf_sid",
		},
	}

	backend.aws.S3GetObjectRequestStub = s3GetObjectRequestStub
	backend.aws.DDBQueryStub = getStateByConfSuccess
	backend.twilio.GetCallStub = getCallSuccess

	backend.aws.DDBDeleteItemStub = func(input *dynamodb.DeleteItemInput) (*dynamodb.DeleteItemOutput, error) {
		return nil, errors.New("damnit")
	}

	backend.logger.ErrorfStub = func(format string, args ...interface{}) {
		a.EqualValues("Error deleting state for call sid %v %v", format)
		a.EqualValues("call_sid", args[0])
		a.EqualValues("damnit", error(args[1].(error)).Error())
	}

	backend.handlersPriv.JiraEnabled = false

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorfCallCount())
}

func TestJIRACreateFailure(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test create JIRA ticket failure
	request := Request{
		APIGatewayProxyRequest: events.APIGatewayProxyRequest{
			Path: "/recordingevent",
			Body: "RecordingStatus=completed&ConferenceSid=conf_sid",
		},
	}

	backend.aws.S3GetObjectRequestStub = s3GetObjectRequestStub
	backend.aws.DDBQueryStub = getStateByConfSuccess
	backend.twilio.GetCallStub = getCallSuccess

	backend.jira.IssueCreateStub = func(issue *goJira.Issue) (*goJira.Issue, *goJira.Response, error) {
		return nil, nil, errors.New("JIRA create failed")
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Error executing handler: ", args[0])
		a.EqualValues("jira.Issue.Create: JIRA create failed", error(args[1].(error)).Error())
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorCallCount())
}

func TestJIRACreateSuccess(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test successful creation of JIRA Ticket
	request := Request{
		APIGatewayProxyRequest: events.APIGatewayProxyRequest{
			Path: "/recordingevent",
			Body: "RecordingStatus=completed&ConferenceSid=conf_sid",
		},
	}

	backend.aws.S3GetObjectRequestStub = s3GetObjectRequestStub
	backend.aws.DDBQueryStub = getStateByConfSuccess
	backend.twilio.GetCallStub = getCallSuccess

	backend.jira.IssueCreateStub = func(issue *goJira.Issue) (*goJira.Issue, *goJira.Response, error) {
		return &goJira.Issue{
			ID: "1234567",
		}, nil, nil
	}

	backend.jira.IssueGetStub = func(issueID string, options *goJira.GetQueryOptions) (*goJira.Issue, *goJira.Response, error) {
		return &goJira.Issue{
			Key: "TICKET-1337",
		}, nil, nil
	}

	backend.logger.DebugfStub = func(format string, args ...interface{}) {
		a.EqualValues("JIRA issue created with Key %v", format)
		a.EqualValues("TICKET-1337", string(args[0].(string)))
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.DebugfCallCount())
	a.EqualValues(0, backend.logger.ErrorCallCount())
}
