package handlers

import (
	"context"
	"encoding/json"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-sdk-go/service/sqs"
	goTwilio "github.com/kevinburke/twilio-go"
	"github.com/pkg/errors"
	"time"
)

// ConferenceLoopMessage contains the data required to be passed between each iteration of the ongoing call loop to keep track of what it needs to do
type ConferenceLoopMessage struct {
	Initiated time.Time `json:"initiated"`
	CallSid   string    `json:"trigger"`
}

var sqsLoopSendDelay int64 = 5
var initialTimeout = 30 * time.Second
var escalateTimeout = 60 * time.Second

func (h *handlers) sqsHandler(ctx context.Context, request events.SQSEvent) []error {
	h.logger.Debug("SQS handler called")

	retErrors := make([]error, 0)

	if len(request.Records) == 0 {
		retErrors = append(retErrors, errors.New("No SQS messages in list"))
		return retErrors
	}

	for _, message := range request.Records {
		msg := &ConferenceLoopMessage{}

		err := json.Unmarshal([]byte(message.Body), msg)
		if err != nil {
			retErrors = append(retErrors, errors.Wrap(err, "Failed to unmarshal SQS message: "))
			h.logger.Debug("Failed to unmarshal SQS message ", message.Body)
			continue
		}

		handlerErr := h.handleSQSRecord(ctx, msg)
		if handlerErr != nil {
			retErrors = append(retErrors, handlerErr)
		}
	}

	return retErrors
}

func (h *handlers) handleSQSRecord(ctx context.Context, message *ConferenceLoopMessage) error {
	var conference *goTwilio.Conference

	if message.CallSid == "" {
		return errors.New("Missing call sid")
	}

	state, err := h.getStateFromCall(message.CallSid)
	if err != nil {
		h.logger.Warn("State not found (this is usually because the call has ended) ", err)
		return nil
	}

	if state.ConferenceSid != "" {
		conference, err = h.twilio.GetConference(ctx, state.ConferenceSid)
		if err != nil {
			h.logger.Errorf("Error retrieving conference %v - %v ", state.ConferenceSid, err)
		} else if conference.Status == "completed" {
			h.logger.Debugf("Conference %v is complete, closing loop", conference.Sid)
			return nil
		}
	} else {
		// If the original caller never made it to the conference (e.g. they hung up before the initial message was finished)
		// Then treat it like the call has finished and clean up because we can't rely on the end-conference handler to do it
		// as the conference never started
		call, err := h.twilio.GetCall(ctx, message.CallSid)
		if err != nil {
			h.logger.Error("Could not retrieve initial call status ", err)
		} else if call.Status != "in-progress" {
			h.logger.Debug("Caller hung up before conference was started, cleaning up")

			h.logger.Debug("Resolving Pagerduty incident")
			err := h.pd.ResolveEvent(h.PagerdutyTwilioIntegrationKey, h.ConferenceName, "Incident Resolved")
			if err != nil {
				h.logger.Error("Error resolving Pagerduty Incident: ", err)
			}

			h.hangupOutgoingCalls(ctx, state.OutgoingCalls)

			err = h.deleteState(state)
			if err != nil {
				h.logger.Error("Could not delete state ", err)
			}

			return nil
		}
	}

	if !state.TeamPickedUp && state.LastEscalation.Add(escalateTimeout).Before(time.Now().UTC()) {
		// Escalate the call to the next escalation level
		outgoingCalls, err := h.dialOnCalls(state.EscalatedLevel+1, generatePickupURL(state.RootURL, state.TriggerCallSid))
		if err != nil {
			h.logger.Error("Failed to escalate call: ", err)
		} else {
			state.Escalated = true
			state.EscalatedLevel = state.EscalatedLevel + 1
			state.LastEscalation = time.Now()
			for _, sid := range outgoingCalls {
				state.OutgoingCalls = append(state.OutgoingCalls, sid)
			}

			err := h.putState(state)
			if err != nil {
				h.logger.Error("Could not put state ", err)
			}
		}
	}

	return h.sendSQSLoop(message)
}

func (h *handlers) sendSQSLoop(message *ConferenceLoopMessage) error {
	bytes, _ := json.Marshal(message)
	msgBody := string(bytes)

	resp, err := h.aws.SendSQSMessage(&sqs.SendMessageInput{
		DelaySeconds: &sqsLoopSendDelay,
		QueueUrl:     &h.SQSQueueURL,
		MessageBody:  &msgBody,
	})

	if err != nil {
		return errors.Wrap(err, "Error sending message for SQS send")
	}

	if resp != nil {
		h.logger.Debug("Send SQS loop message with ID ", *resp.MessageId)
	}

	return nil
}
