package handlers

import (
	"context"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/sqs"
	goTwilio "github.com/kevinburke/twilio-go"
	pdEvents "github.com/marcw/pagerduty"
	"github.com/pkg/errors"
	"github.com/stretchr/testify/assert"
	"net/url"
	"strings"
	"testing"
)

var newRequest = Request{
	SQSEvent: events.SQSEvent{
		Records: []events.SQSMessage{
			events.SQSMessage{
				Body: `{"trigger":"initial_call_sid"}`,
			},
		},
	},
}

var getConferenceSuccess = func(ctx context.Context, sid string) (*goTwilio.Conference, error) {
	return &goTwilio.Conference{
		Status: "completed",
	}, nil
}

var getIncompleteConferenceSuccess = func(ctx context.Context, sid string) (*goTwilio.Conference, error) {
	return &goTwilio.Conference{
		Status: "in-progress",
	}, nil
}

func TestEmptyRecordSet(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test empty record set
	request := Request{
		SQSEvent: events.SQSEvent{
			Records: make([]events.SQSMessage, 0),
		},
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Error executing SQS handler: ", args[0])
		a.EqualValues("No SQS messages in list", error(args[1].(error)).Error())
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorCallCount())
}

func TestInvalidJSON(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test invalid JSON
	request := Request{
		SQSEvent: events.SQSEvent{
			Records: []events.SQSMessage{
				events.SQSMessage{
					Body: "this is not json",
				},
			},
		},
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Error executing SQS handler: ", args[0])
		a.True(strings.HasPrefix(error(args[1].(error)).Error(), "Failed to unmarshal SQS message:"), "error message should start with Failed to unmarshal SQS message: - got `%v`", error(args[1].(error)).Error())
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorCallCount())
}

func TestMissingCallSid(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test missing call sid
	request := Request{
		SQSEvent: events.SQSEvent{
			Records: []events.SQSMessage{
				events.SQSMessage{
					Body: `{}`,
				},
			},
		},
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Error executing SQS handler: ", args[0])
		a.EqualValues("Missing call sid", args[1].(error).Error())
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorCallCount())
}

func TestFailedToGetState(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test failure to get state
	request := newRequest
	backend.aws.DDBGetItemStub = func(input *dynamodb.GetItemInput) (*dynamodb.GetItemOutput, error) {
		return nil, errors.New("Failed to get state")
	}

	backend.logger.WarnStub = func(args ...interface{}) {
		a.EqualValues("State not found (this is usually because the call has ended) ", args[0])
		a.EqualValues("Failed to get state", args[1].(error).Error())
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.WarnCallCount())
}

func TestErrorRetrievingConference(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test failure to get conference
	request := newRequest
	backend.aws.DDBGetItemStub = getStateWithConfSuccess
	backend.pd.ListOnCallsStub = onCallList
	backend.twilio.GetConferenceStub = func(ctx context.Context, sid string) (*goTwilio.Conference, error) {
		return nil, errors.New("conf is busted")
	}

	backend.logger.ErrorfStub = func(format string, args ...interface{}) {
		a.EqualValues("Error retrieving conference %v - %v ", format)
		a.EqualValues("conference_sid", args[0])
		a.EqualValues("conf is busted", args[1].(error).Error())
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorfCallCount())
}

func TestFailureToDialOnCall(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test failure to dial oncall
	request := newRequest
	backend.aws.DDBGetItemStub = getStateWithConfSuccess
	backend.pd.ListOnCallsStub = onCallList
	backend.pd.ListTeamUsersStub = teamUsers
	backend.twilio.GetConferenceStub = getIncompleteConferenceSuccess
	backend.twilio.CallStub = func(from string, to string, u *url.URL) (*goTwilio.Call, error) {
		return nil, errors.New("could not pick up the phone")
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Failed to escalate call: ", args[0])
		a.EqualValues("Zero successful dials", args[1].(error).Error())
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorCallCount())
}

func TestErrorPuttingEscalatedState(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test failure to put escalated state
	request := newRequest
	backend.aws.DDBGetItemStub = getStateWithConfSuccess
	backend.pd.ListOnCallsStub = onCallList
	backend.pd.ListTeamUsersStub = teamUsers
	backend.twilio.GetConferenceStub = getIncompleteConferenceSuccess
	backend.twilio.CallStub = createCall
	backend.aws.DDBPutItemStub = func(input *dynamodb.PutItemInput) (*dynamodb.PutItemOutput, error) {
		return nil, errors.New("DDB is broke again")
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Could not put state ", args[0])
		a.EqualValues("DDB is broke again", args[1].(error).Error())
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorCallCount())
}

func TestSuccessfulMessage(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test success
	request := newRequest
	backend.aws.DDBGetItemStub = getStateWithConfSuccess
	backend.pd.ListOnCallsStub = onCallList
	backend.pd.ListTeamUsersStub = teamUsers
	backend.twilio.GetConferenceStub = getIncompleteConferenceSuccess
	backend.twilio.CallStub = createCall

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(0, backend.logger.ErrorCallCount())
}

func TestSuccessfulConfCompleteMessage(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test success
	request := newRequest
	backend.aws.DDBGetItemStub = getStateWithConfSuccess
	backend.pd.ListOnCallsStub = onCallList
	backend.pd.ListTeamUsersStub = teamUsers
	backend.twilio.GetConferenceStub = getConferenceSuccess
	backend.twilio.CallStub = createCall

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(0, backend.logger.ErrorCallCount())
}

func TestSQSMessageSend(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)
	message := &ConferenceLoopMessage{}

	backend.aws.SendSQSMessageStub = func(input *sqs.SendMessageInput) (*sqs.SendMessageOutput, error) {
		return nil, errors.New("SQS is busted")
	}

	err = backend.handlersPriv.sendSQSLoop(message)
	a.NotNil(err)
	a.EqualValues("Error sending message for SQS send: SQS is busted", err.Error())

	backend.aws.SendSQSMessageStub = func(input *sqs.SendMessageInput) (*sqs.SendMessageOutput, error) {
		messageID := "message_id"
		return &sqs.SendMessageOutput{
			MessageId: &messageID,
		}, nil
	}

	backend.logger.DebugStub = func(args ...interface{}) {
		a.EqualValues("Send SQS loop message with ID ", args[0])
		a.EqualValues("message_id", args[1])
	}

	err = backend.handlersPriv.sendSQSLoop(message)
	a.Nil(err)
	a.EqualValues(1, backend.logger.DebugCallCount())
}

func TestEmptyConferenceGetCallError(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test failure to delete state
	request := newRequest
	backend.aws.DDBGetItemStub = getEscalatedStateSuccess
	backend.pd.ListOnCallsStub = onCallList
	backend.twilio.GetConferenceStub = getConferenceSuccess
	backend.twilio.GetCallStub = func(ctx context.Context, sid string) (*goTwilio.Call, error) {
		return nil, errors.New("getcall error")
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Could not retrieve initial call status ", args[0])
		a.EqualValues("getcall error", args[1].(error).Error())
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorCallCount())
}

func TestEmptyConferenceResolvePDError(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test failure to delete state
	request := newRequest
	backend.aws.DDBGetItemStub = getEscalatedStateSuccess
	backend.pd.ListOnCallsStub = onCallList
	backend.twilio.GetConferenceStub = getConferenceSuccess
	backend.twilio.GetCallStub = getCallSuccess

	backend.pd.SubmitEventStub = func(event *pdEvents.Event) (response *pdEvents.Response, statusCode int, err error) {
		return nil, 0, errors.New("PD errrrr")
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Error resolving Pagerduty Incident: ", args[0])
		a.EqualValues("Failed to submit PD event: PD errrrr", args[1].(error).Error())
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorCallCount())
}

func TestEmptyConferenceDeleteStateError(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test failure to delete state
	request := newRequest
	backend.aws.DDBGetItemStub = getStateSuccess
	backend.pd.ListOnCallsStub = onCallList
	backend.twilio.GetConferenceStub = getConferenceSuccess
	backend.twilio.GetCallStub = getCallSuccess
	backend.pd.SubmitEventStub = func(event *pdEvents.Event) (response *pdEvents.Response, statusCode int, err error) {
		return nil, 200, nil
	}

	backend.aws.DDBDeleteItemStub = func(input *dynamodb.DeleteItemInput) (*dynamodb.DeleteItemOutput, error) {
		return nil, errors.New("Error deleting things from dynamodb")
	}

	backend.logger.ErrorStub = func(args ...interface{}) {
		a.EqualValues("Could not delete state ", args[0])
		a.EqualValues("Error deleting things from dynamodb", args[1].(error).Error())
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(1, backend.logger.ErrorCallCount())
}

func TestEmptyConferenceSuccess(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	ctx := context.TODO()

	// Test failure to delete state
	request := newRequest
	backend.aws.DDBGetItemStub = getEscalatedStateSuccess
	backend.pd.ListOnCallsStub = onCallList
	backend.twilio.GetConferenceStub = getConferenceSuccess
	backend.twilio.GetCallStub = func(ctx context.Context, sid string) (*goTwilio.Call, error) {
		return &goTwilio.Call{
			Sid:    "ABC123",
			Status: "ringing",
			From:   "+441234567890",
			To:     "+1975638394",
		}, nil
	}

	backend.pd.SubmitEventStub = func(event *pdEvents.Event) (response *pdEvents.Response, statusCode int, err error) {
		return nil, 200, nil
	}

	_, err = backend.handlersPub.Handle(ctx, request)
	a.Nil(err)
	a.EqualValues(0, backend.logger.ErrorCallCount())
	a.EqualValues(1, backend.twilio.HangupCallCount())
	a.EqualValues(1, backend.pd.SubmitEventCallCount())
	a.EqualValues(1, backend.aws.DDBDeleteItemCallCount())
}
