package handlers

import (
	"fmt"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/pkg/errors"
	"strconv"
	"time"
)

// State contains all the information required for various parts of the system to function during the progress of a call
type State struct {
	Initiated      time.Time
	TriggerCallSid string
	ConferenceSid  string
	OutgoingCalls  []*string
	Escalated      bool
	EscalatedLevel int
	LastEscalation time.Time
	RootURL        string
	TeamPickedUp   bool
}

const timeLayout = "2006-01-02T15:04:05Z"

func mapStateFromAttributes(attributes map[string]*dynamodb.AttributeValue) (*State, error) {
	state := &State{}

	if val, ok := attributes["initiated"]; ok {
		initiated, err := time.Parse(timeLayout, *val.S)
		if err != nil {
			return nil, err
		}
		state.Initiated = initiated.UTC()
	}

	if val, ok := attributes["call_sid"]; ok {
		state.TriggerCallSid = *val.S
	}

	if val, ok := attributes["conference_sid"]; ok {
		state.ConferenceSid = *val.S
	}

	if val, ok := attributes["outgoing_calls"]; ok {
		state.OutgoingCalls = val.SS
	}

	if val, ok := attributes["escalated"]; ok {
		state.Escalated = *val.BOOL
	}

	if val, ok := attributes["escalated_level"]; ok {
		valN, err := strconv.Atoi(*val.N)
		if err != nil {
			return nil, err
		}
		state.EscalatedLevel = valN
	}

	if val, ok := attributes["last_escalation"]; ok {
		lastEscalation, err := time.Parse(timeLayout, *val.S)
		if err != nil {
			return nil, err
		}
		state.LastEscalation = lastEscalation.UTC()
	}

	if val, ok := attributes["root_url"]; ok {
		state.RootURL = *val.S
	}

	if val, ok := attributes["team_picked_up"]; ok {
		state.TeamPickedUp = *val.BOOL
	}

	return state, nil
}

func mapAttributesFromState(state *State) map[string]*dynamodb.AttributeValue {
	attributes := make(map[string]*dynamodb.AttributeValue, 0)

	initiated := state.Initiated.Format(timeLayout)
	attributes["initiated"] = &dynamodb.AttributeValue{S: &initiated}

	if state.TriggerCallSid != "" {
		attributes["call_sid"] = &dynamodb.AttributeValue{S: &state.TriggerCallSid}
	}
	if state.ConferenceSid != "" {
		attributes["conference_sid"] = &dynamodb.AttributeValue{S: &state.ConferenceSid}
	}
	if state.OutgoingCalls != nil {
		attributes["outgoing_calls"] = &dynamodb.AttributeValue{SS: state.OutgoingCalls}
	}
	if state.RootURL != "" {
		attributes["root_url"] = &dynamodb.AttributeValue{S: &state.RootURL}
	}

	lastEscalation := state.LastEscalation.Format(timeLayout)
	attributes["last_escalation"] = &dynamodb.AttributeValue{S: &lastEscalation}

	attributes["escalated"] = &dynamodb.AttributeValue{BOOL: &state.Escalated}

	escalatedLevel := strconv.Itoa(state.EscalatedLevel)
	attributes["escalated_level"] = &dynamodb.AttributeValue{N: &escalatedLevel}

	attributes["team_picked_up"] = &dynamodb.AttributeValue{BOOL: &state.TeamPickedUp}

	return attributes
}

func (h *handlers) getStateFromCall(callSid string) (*State, error) {
	input := &dynamodb.GetItemInput{
		TableName: &h.DynamoDBTableName,
		Key: map[string]*dynamodb.AttributeValue{
			"call_sid": &dynamodb.AttributeValue{
				S: &callSid,
			},
		},
	}
	output, err := h.aws.DDBGetItem(input)
	if err != nil {
		return nil, err
	}

	if output.Item == nil {
		return nil, errors.New("Empty item returned")
	}

	return mapStateFromAttributes(output.Item)
}

func (h *handlers) getStateFromConference(confSid string) (*State, error) {
	filter := "conference_sid = :conf_sid"
	index := "ConferenceSidIndex"

	input := &dynamodb.QueryInput{
		TableName:              &h.DynamoDBTableName,
		IndexName:              &index,
		KeyConditionExpression: &filter,
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":conf_sid": &dynamodb.AttributeValue{S: &confSid},
		},
	}
	output, err := h.aws.DDBQuery(input)

	if err != nil {
		return nil, err
	}

	if len(output.Items) == 0 {
		return nil, errors.New("No Items returned")
	}

	if len(output.Items) > 1 {
		return nil, fmt.Errorf("Duplicate entries for conference %v found", confSid)
	}

	return mapStateFromAttributes(output.Items[0])
}

func (h *handlers) putState(state *State) error {
	item := mapAttributesFromState(state)

	input := &dynamodb.PutItemInput{
		TableName: &h.DynamoDBTableName,
		Item:      item,
	}

	_, err := h.aws.DDBPutItem(input)

	return err
}

func (h *handlers) deleteState(state *State) error {
	input := &dynamodb.DeleteItemInput{
		TableName: &h.DynamoDBTableName,
		Key: map[string]*dynamodb.AttributeValue{
			"call_sid": &dynamodb.AttributeValue{
				S: &state.TriggerCallSid,
			},
		},
	}
	_, err := h.aws.DDBDeleteItem(input)

	return err
}
