package handlers

import (
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/pkg/errors"
	"github.com/stretchr/testify/assert"
	"strconv"
	"testing"
	"time"
)

var getStateSuccess = func(input *dynamodb.GetItemInput) (*dynamodb.GetItemOutput, error) {
	callSid := "call_sid"
	escalated := false
	now := time.Now().UTC().Format(timeLayout)
	escalatedLevel := "1"

	return &dynamodb.GetItemOutput{
		Item: map[string]*dynamodb.AttributeValue{
			"call_sid":        &dynamodb.AttributeValue{S: &callSid},
			"escalated":       &dynamodb.AttributeValue{BOOL: &escalated},
			"last_escalation": &dynamodb.AttributeValue{S: &now},
			"escalated_level": &dynamodb.AttributeValue{N: &escalatedLevel},
		},
	}, nil
}

var getEscalatedStateSuccess = func(input *dynamodb.GetItemInput) (*dynamodb.GetItemOutput, error) {
	callSid := "call_sid"
	escalated := true
	now := time.Now().UTC().Format(timeLayout)
	escalatedLevel := "2"

	return &dynamodb.GetItemOutput{
		Item: map[string]*dynamodb.AttributeValue{
			"call_sid":        &dynamodb.AttributeValue{S: &callSid},
			"escalated":       &dynamodb.AttributeValue{BOOL: &escalated},
			"outgoing_calls":  &dynamodb.AttributeValue{SS: []*string{&callSid}},
			"last_escalation": &dynamodb.AttributeValue{S: &now},
			"escalated_level": &dynamodb.AttributeValue{N: &escalatedLevel},
		},
	}, nil
}

var getStateWithConfSuccess = func(input *dynamodb.GetItemInput) (*dynamodb.GetItemOutput, error) {
	callSid := "call_sid"
	conferenceSid := "conference_sid"
	escalated := false
	now := time.Now().Add(-1 * time.Minute).UTC().Format(timeLayout)
	escalatedLevel := "1"

	return &dynamodb.GetItemOutput{
		Item: map[string]*dynamodb.AttributeValue{
			"call_sid":        &dynamodb.AttributeValue{S: &callSid},
			"conference_sid":  &dynamodb.AttributeValue{S: &conferenceSid},
			"escalated":       &dynamodb.AttributeValue{BOOL: &escalated},
			"last_escalation": &dynamodb.AttributeValue{S: &now},
			"escalated_level": &dynamodb.AttributeValue{N: &escalatedLevel},
		},
	}, nil
}

var getStateByConfSuccess = func(input *dynamodb.QueryInput) (*dynamodb.QueryOutput, error) {
	escalated := true
	callSid := "call_sid"
	now := time.Now().UTC().Format(timeLayout)
	escalatedLevel := "2"

	return &dynamodb.QueryOutput{
		Items: []map[string]*dynamodb.AttributeValue{
			map[string]*dynamodb.AttributeValue{
				"call_sid":        &dynamodb.AttributeValue{S: &callSid},
				"escalated":       &dynamodb.AttributeValue{BOOL: &escalated},
				"last_escalation": &dynamodb.AttributeValue{S: &now},
				"escalated_level": &dynamodb.AttributeValue{N: &escalatedLevel},
			},
		},
	}, nil
}

func TestMappings(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	callSid := "call_sid"
	initiated := "2018-06-08T12:15:38Z"
	conferenceSid := "conference_sid"
	outgoingCalls := []*string{&callSid}
	escalated := true
	teamPickedUp := false
	rootURL := "root_url"
	escalatedLevel := "1"

	// Test erroring timestamp parse
	badTimestamp := "abc123"
	attributes := map[string]*dynamodb.AttributeValue{
		"initiated": &dynamodb.AttributeValue{S: &badTimestamp},
	}

	state, err := mapStateFromAttributes(attributes)
	a.NotNil(err)
	a.IsType(&time.ParseError{}, err)

	attributes = map[string]*dynamodb.AttributeValue{
		"last_escalation": &dynamodb.AttributeValue{S: &badTimestamp},
	}

	state, err = mapStateFromAttributes(attributes)
	a.NotNil(err)
	a.IsType(&time.ParseError{}, err)

	// Test erroring number parse
	badNumber := "abc123"
	attributes = map[string]*dynamodb.AttributeValue{
		"escalated_level": &dynamodb.AttributeValue{N: &badNumber},
	}

	state, err = mapStateFromAttributes(attributes)
	a.NotNil(err)
	a.IsType(&strconv.NumError{}, err)

	attributes["initiated"] = &dynamodb.AttributeValue{S: &initiated}
	attributes["call_sid"] = &dynamodb.AttributeValue{S: &callSid}
	attributes["conference_sid"] = &dynamodb.AttributeValue{S: &conferenceSid}
	attributes["outgoing_calls"] = &dynamodb.AttributeValue{SS: outgoingCalls}
	attributes["escalated"] = &dynamodb.AttributeValue{BOOL: &escalated}
	attributes["root_url"] = &dynamodb.AttributeValue{S: &rootURL}
	attributes["team_picked_up"] = &dynamodb.AttributeValue{BOOL: &teamPickedUp}
	attributes["last_escalation"] = &dynamodb.AttributeValue{S: &initiated}
	attributes["escalated_level"] = &dynamodb.AttributeValue{N: &escalatedLevel}

	state, err = mapStateFromAttributes(attributes)
	a.Nil(err)
	testTime, err := time.Parse("2006-01-02T15:04:05Z", "2018-06-08T12:15:38Z")
	a.Nil(err)

	a.EqualValues(testTime, state.Initiated)
	a.EqualValues("call_sid", state.TriggerCallSid)
	a.EqualValues("conference_sid", state.ConferenceSid)
	a.EqualValues("call_sid", *state.OutgoingCalls[0])
	a.EqualValues(true, state.Escalated)
	a.EqualValues("root_url", state.RootURL)
	a.EqualValues(false, state.TeamPickedUp)

	// Test mapping back the other way
	attributes = mapAttributesFromState(state)

	a.EqualValues("2018-06-08T12:15:38Z", *attributes["initiated"].S)
	a.EqualValues("call_sid", *attributes["call_sid"].S)
	a.EqualValues("conference_sid", *attributes["conference_sid"].S)
	a.EqualValues("call_sid", *attributes["outgoing_calls"].SS[0])
	a.EqualValues(true, *attributes["escalated"].BOOL)
	a.EqualValues("root_url", *attributes["root_url"].S)
	a.EqualValues(false, *attributes["team_picked_up"].BOOL)
}

func TestGetState(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	// Test Erroring DDB Call
	backend.aws.DDBGetItemStub = func(input *dynamodb.GetItemInput) (*dynamodb.GetItemOutput, error) {
		return nil, errors.New("oopsie woopsie")
	}

	state, err := backend.handlersPriv.getStateFromCall("call_sid")
	a.Nil(state)
	a.NotNil(err)
	a.EqualValues("oopsie woopsie", err.Error())

	// Test empty Item
	backend.aws.DDBGetItemStub = func(input *dynamodb.GetItemInput) (*dynamodb.GetItemOutput, error) {
		return &dynamodb.GetItemOutput{}, nil
	}

	state, err = backend.handlersPriv.getStateFromCall("call_sid")
	a.Nil(state)
	a.NotNil(err)
	a.EqualValues("Empty item returned", err.Error())

	// Test successful get
	backend.aws.DDBGetItemStub = func(input *dynamodb.GetItemInput) (*dynamodb.GetItemOutput, error) {
		a.EqualValues("call_sid", *input.Key["call_sid"].S)
		return getStateSuccess(input)
	}
	state, err = backend.handlersPriv.getStateFromCall("call_sid")
	a.Nil(err)
	a.NotNil(state)
	a.EqualValues("call_sid", state.TriggerCallSid)

	// Test Get by Conference Sid
	// Test erroring DDB call
	backend.aws.DDBQueryStub = func(input *dynamodb.QueryInput) (*dynamodb.QueryOutput, error) {
		return nil, errors.New("fucky wucky")
	}
	state, err = backend.handlersPriv.getStateFromConference("conf_sid")
	a.Nil(state)
	a.NotNil(err)
	a.EqualValues("fucky wucky", err.Error())

	// Test empty Items
	backend.aws.DDBQueryStub = func(input *dynamodb.QueryInput) (*dynamodb.QueryOutput, error) {
		return &dynamodb.QueryOutput{}, nil
	}
	state, err = backend.handlersPriv.getStateFromConference("conf_sid")
	a.Nil(state)
	a.NotNil(err)
	a.EqualValues("No Items returned", err.Error())

	// Test duplicate Items
	backend.aws.DDBQueryStub = func(input *dynamodb.QueryInput) (*dynamodb.QueryOutput, error) {
		return &dynamodb.QueryOutput{
			Items: []map[string]*dynamodb.AttributeValue{
				map[string]*dynamodb.AttributeValue{},
				map[string]*dynamodb.AttributeValue{},
			},
		}, nil
	}
	state, err = backend.handlersPriv.getStateFromConference("conf_sid")
	a.Nil(state)
	a.NotNil(err)
	a.EqualValues("Duplicate entries for conference conf_sid found", err.Error())

	// Test successful get
	backend.aws.DDBQueryStub = func(input *dynamodb.QueryInput) (*dynamodb.QueryOutput, error) {
		a.EqualValues("conf_sid", *input.ExpressionAttributeValues[":conf_sid"].S)
		return getStateByConfSuccess(input)
	}
	state, err = backend.handlersPriv.getStateFromConference("conf_sid")
	a.Nil(err)
	a.NotNil(state)
	a.EqualValues(true, state.Escalated)
}

func TestPutState(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	// We basically just call the underlying put and return the error so there's not a whole lot to test
	backend.aws.DDBPutItemStub = func(input *dynamodb.PutItemInput) (*dynamodb.PutItemOutput, error) {
		a.EqualValues("conference_sid", *input.Item["conference_sid"].S)
		return nil, nil
	}

	state := &State{
		ConferenceSid: "conference_sid",
	}

	err = backend.handlersPriv.putState(state)
	a.Nil(err)
	a.EqualValues(1, backend.aws.DDBPutItemCallCount())
}

func TestDeleteState(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	backend, err := createMockBackend()
	a.Nil(err)

	// We basically just call the underlying put and return the error so there's not a whole lot to test
	backend.aws.DDBDeleteItemStub = func(input *dynamodb.DeleteItemInput) (*dynamodb.DeleteItemOutput, error) {
		a.EqualValues("call_sid", *input.Key["call_sid"].S)
		return nil, nil
	}

	state := &State{
		TriggerCallSid: "call_sid",
	}

	err = backend.handlersPriv.deleteState(state)
	a.Nil(err)
	a.EqualValues(1, backend.aws.DDBDeleteItemCallCount())
}
