package backend

import (
	"net/http"
	"net/url"

	goJira "github.com/andygrunwald/go-jira"
)

// Client redefines goJira.Client to allow custom methods and easier data mocking for tests.
// If this interface changes, counterfeiter must be re-run using `make mocks`
type Client interface {
	GetBaseURL() url.URL
	IssueGet(issueID string, options *goJira.GetQueryOptions) (*goJira.Issue, *goJira.Response, error)
	IssueCreate(issue *goJira.Issue) (*goJira.Issue, *goJira.Response, error)
}

type client struct {
	jira *goJira.Client
}

// New calls the NewClient function from the base jira package
func New(httpClient *http.Client, hostName string) (Client, error) {
	goJiraClient, err := goJira.NewClient(httpClient, "https://"+hostName+"/")
	if err != nil {
		return nil, err
	}
	return &client{jira: goJiraClient}, nil
}

// GetConfig returns the configuration object for this client
func (c *client) GetConfig() *client {
	return c
}

// GetBaseURL calls the underlying client's GetBaseURL function
func (c *client) GetBaseURL() url.URL {
	return c.jira.GetBaseURL()
}

// IssueGet calls the underlying client's Issue.Get function
func (c *client) IssueGet(issueID string, options *goJira.GetQueryOptions) (*goJira.Issue, *goJira.Response, error) {
	return c.jira.Issue.Get(issueID, options)
}

// IssueCreate calls the underlying client's Issue.Create function
func (c *client) IssueCreate(issue *goJira.Issue) (*goJira.Issue, *goJira.Response, error) {
	return c.jira.Issue.Create(issue)
}
