package jira

import (
	"code.justin.tv/event-engineering/goldengate/pkg/jira/backend"
	logging "code.justin.tv/event-engineering/goldengate/pkg/logging/backend"
	"github.com/pkg/errors"

	goJira "github.com/andygrunwald/go-jira"
	tcontainer "github.com/trivago/tgo/tcontainer"
	"io/ioutil"
)

// New generates a new Jira Client with the supplied backend client
func New(jiraClient backend.Client, logger logging.Logger) *Client {
	return &Client{
		jira:   jiraClient,
		logger: logger,
	}
}

// Client is a JIRA client containing methods for use in the goldengate system
type Client struct {
	jira   backend.Client
	logger logging.Logger
}

// CreateTicket makes a new Jira Task
func (c *Client) CreateTicket(jiraTicket *Ticket) error {
	customfields := tcontainer.NewMarshalMap()

	customfields["customfield_17901"] = jiraTicket.Caller
	customfields["customfield_17902"] = jiraTicket.ConferenceSid
	customfields["customfield_17903"] = jiraTicket.RecordingSid
	customfields["customfield_17905"] = jiraTicket.RecordingURL

	issue, resp, err := c.jira.IssueCreate(&goJira.Issue{
		Fields: &goJira.IssueFields{
			Type: goJira.IssueType{
				Name: jiraTicket.IssueType,
			},
			Project: goJira.Project{
				Key: jiraTicket.ProjectKey,
			},
			Summary:     jiraTicket.Summary,
			Description: jiraTicket.Description,
			Unknowns:    customfields,
		},
	})

	if err != nil {
		if resp != nil && resp.Body != nil {
			bytes, readErr := ioutil.ReadAll(resp.Body)
			if readErr != nil {
				c.logger.Error("Error reading response body ", readErr)
			} else {
				c.logger.Warn(string(bytes))
			}
		}

		return errors.Wrap(err, "jira.Issue.Create")
	}

	issue, _, err = c.jira.IssueGet(issue.ID, nil)
	if err != nil {
		return errors.Wrap(err, "jira.Issue.Get")
	}

	c.mapIssueFromJIRA(jiraTicket, issue)
	return nil
}

func (c *Client) mapIssueFromJIRA(ticket *Ticket, issue *goJira.Issue) {
	ticket.TicketKey = issue.Key

	baseURL := c.jira.GetBaseURL()
	host := baseURL.Hostname()
	ticket.URL = "https://" + host + "/browse/" + issue.Key

	if issue.Fields != nil {
		ticket.Status = issue.Fields.Status.Name
		ticket.IssueType = issue.Fields.Type.Name
		ticket.ProjectKey = issue.Fields.Project.Key
		ticket.Summary = issue.Fields.Summary
		ticket.Description = issue.Fields.Description
	}
}

// Ticket is the data needed to make a new task.
type Ticket struct {
	TicketKey     string
	ProjectKey    string
	Summary       string
	Description   string
	IssueType     string
	URL           string
	Status        string
	Caller        string
	ConferenceSid string
	RecordingSid  string
	RecordingURL  string
}
