package jira

import (
	"bytes"
	"code.justin.tv/event-engineering/goldengate/pkg/jira/backend/backendfakes"
	loggingfakes "code.justin.tv/event-engineering/goldengate/pkg/logging/backend/backendfakes"
	"errors"
	"github.com/stretchr/testify/assert"
	"io"
	"net/http"
	"net/url"
	"strings"
	"testing"

	goJira "github.com/andygrunwald/go-jira"
	tcontainer "github.com/trivago/tgo/tcontainer"
)

func TestIsMatch(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	fakeClient := new(backendfakes.FakeClient)
	fakeLogger := new(loggingfakes.FakeLogger)

	jiraClient := New(fakeClient, fakeLogger)
	a.NotNil(jiraClient)

	testTicket := &Ticket{
		ProjectKey:    "BROAD",
		Summary:       "Test Summary",
		Description:   "Test Description",
		IssueType:     "Event Incident",
		Caller:        "1234567890",
		ConferenceSid: "ACB123DEF456",
		RecordingSid:  "GHI789JKL",
		RecordingURL:  "https://www.example.com",
	}

	customfields := tcontainer.NewMarshalMap()
	customfields["customfield_17901"] = []map[string]string{{"value": testTicket.Caller}}
	customfields["customfield_17902"] = []map[string]string{{"value": testTicket.ConferenceSid}}
	customfields["customfield_17903"] = []map[string]string{{"value": testTicket.RecordingSid}}
	customfields["customfield_17905"] = []map[string]string{{"value": testTicket.RecordingURL}}

	goJiraIssues := make(map[string]*goJira.Issue)

	goJiraIssues["12345"] = &goJira.Issue{
		Key: "BROAD1337",
		Fields: &goJira.IssueFields{
			Status: &goJira.Status{
				Name: "Open",
			},
			Type: goJira.IssueType{
				Name: testTicket.IssueType,
			},
			Project: goJira.Project{
				Key: testTicket.ProjectKey,
			},
			Summary:     testTicket.Summary,
			Description: testTicket.Description,
			Unknowns:    customfields,
		},
	}

	fakeClient.GetBaseURLStub = func() url.URL {
		url, _ := url.Parse("https://www.example.com")
		return *url
	}

	fakeClient.IssueCreateStub = func(issue *goJira.Issue) (*goJira.Issue, *goJira.Response, error) {
		issue.ID = "12345"

		mapValue, err := issue.Fields.Unknowns.String("customfield_17901")
		a.Nil(err)
		a.EqualValues("1234567890", mapValue)

		mapValue, err = issue.Fields.Unknowns.String("customfield_17902")
		a.Nil(err)
		a.EqualValues("ACB123DEF456", mapValue)

		mapValue, err = issue.Fields.Unknowns.String("customfield_17903")
		a.Nil(err)
		a.EqualValues("GHI789JKL", mapValue)

		mapValue, err = issue.Fields.Unknowns.String("customfield_17905")
		a.Nil(err)
		a.EqualValues("https://www.example.com", mapValue)

		return issue, &goJira.Response{}, nil
	}

	fakeClient.IssueGetStub = func(issueID string, options *goJira.GetQueryOptions) (*goJira.Issue, *goJira.Response, error) {
		if issue, ok := goJiraIssues[issueID]; ok {
			return issue, nil, nil
		}
		return nil, nil, errors.New("No issue found")
	}

	// Test successful create
	err := jiraClient.CreateTicket(testTicket)

	a.Nil(err)
	a.Equal("BROAD1337", testTicket.TicketKey)
	a.Equal("BROAD", testTicket.ProjectKey)
	a.Equal("Test Summary", testTicket.Summary)
	a.Equal("Test Description", testTicket.Description)
	a.Equal("Event Incident", testTicket.IssueType)
	a.Equal("1234567890", testTicket.Caller)
	a.Equal("ACB123DEF456", testTicket.ConferenceSid)
	a.Equal("GHI789JKL", testTicket.RecordingSid)
	a.Equal("https://www.example.com", testTicket.RecordingURL)
	a.Equal("https://www.example.com/browse/BROAD1337", testTicket.URL)

	// Test erroring create (error on issue get)
	fakeClient.IssueGetStub = func(issueID string, options *goJira.GetQueryOptions) (*goJira.Issue, *goJira.Response, error) {
		return nil, nil, errors.New("error retrieving issue")
	}

	err = jiraClient.CreateTicket(testTicket)
	a.NotNil(err)
	a.True(strings.HasPrefix(err.Error(), "jira.Issue.Get:"), "error message should start with jira.Issue.Get: - got `%v`", err.Error())

	// Test erroring create (error on issue create)
	fakeClient.IssueCreateStub = func(issue *goJira.Issue) (*goJira.Issue, *goJira.Response, error) {
		return nil, &goJira.Response{
			Response: &http.Response{
				Body: nopCloser{bytes.NewBufferString("ERROR")},
			},
		}, errors.New("error creating issue")
	}

	err = jiraClient.CreateTicket(testTicket)
	a.NotNil(err)
	a.True(strings.HasPrefix(err.Error(), "jira.Issue.Create:"), "error message should start with jira.Issue.Create: - got `%v`", err.Error())

	// Test erroring create (error on reading response body)
	fakeClient.IssueCreateStub = func(issue *goJira.Issue) (*goJira.Issue, *goJira.Response, error) {
		return nil, &goJira.Response{
			Response: &http.Response{
				Body: nopErrCloser{},
			},
		}, errors.New("error creating issue")
	}

	err = jiraClient.CreateTicket(testTicket)
	a.NotNil(err)
	a.True(strings.HasPrefix(err.Error(), "jira.Issue.Create:"), "error message should start with jira.Issue.Create: - got `%v`", err.Error())

}

// Noop reader closer
type nopCloser struct {
	io.Reader
}

func (nopCloser) Close() error { return nil }

// Erroring reader closer
type nopErrCloser struct{}

func (nopErrCloser) Read(p []byte) (int, error) {
	return 0, errors.New("can't read connection closed or something")
}
func (nopErrCloser) Close() error { return errors.New("can't close connection closed or something") }
